<?php

/**
 * Class Glb_Collection
 * An advanced way to store / update / iterate arrays of values,
 * with a lot of interesting helpers functions that should normally work
 */
class Glb_Collection implements Serializable, Iterator, JsonSerializable
{

    private $position = 0;                         // scroll current position
    //protected $columns = array();                // columns configured
    //protected $key_column = null;                // column used for keying the collection
    protected $items = [];                         // items of this collection
    protected $keys = [];                          // keys of this collection, only for

    /**
     * Glb_Collection Constructor : creates a new Glb_Collection killer object
     * $items (optional) array|object|ArrayObject : array of items to store in this collection
     * $columns (optional) array : array of columns definitions (Glb_Column).
     *      Will be used for validating data or displaying html
     *
     * and the key_column if you want the collection to be keyed
     * if columns are not set, the columns definitions will be ignored
     * and the output function for the entity will not be availbale
     */
    function __construct($items = []/*, $columns = null, $key_column = null, $check_value = false*/) {
        if (is_object($items)) { $items = (array)$items; }
        $this->items = $items;
        $this->keys = array_keys($items);
        $this->position = 0;
    }

    // --------------------------------
    // Iterator implementation functions

    public function current() {
        return $this->items[$this->key()];
        //return $this->items[array_search($this->key(), $this->keys)];
    }

    public function next() {
        $this->position++;
    }

    public function key() {
        return $this->keys[$this->position];
    }

    public function valid() {
        return isset($this->keys[$this->position]);
    }

    public function rewind() {
        $this->position = 0;
    }

    // --------------------------------
    public function __toString() {
        return print_r($this->items, true);
    }

    public function to_array($recursive = false) {
        if (!$recursive) {
            return $this->items;
        } else {
            $result = [];
            foreach($this->items as $key => $value) {
                if (method_exists($value, 'to_array')) {
                    $result[$key] = $value->to_array(true);
                } else {
                    $result[$key] = $value;
                }
            }
            return $result;
        }
    }

    public function keys() {
        return $this->keys;
    }

    public function values() {
        return array_values($this->items);
    }

    /**
     * @function reverse
     * Reverse items, first will be last (and vice versa)
     * Updates the current collection and returns $this
     * @return Glb_Collection $this
     */
    public function reverse() {
        if (!$this->is_associative()) {
            $this->items = array_reverse($this->items, false);
        } else {
            $this->items = array_reverse($this->items, true);
        }
        $this->keys = array_keys($this->items);
        $this->rewind();
        return $this;
    }


    /**
     * @function clear
     * Magically clear the items of this collection
     * Updates the current collection and returns $this
     * @return Glb_Collection $this
     */
    public function clear() {
        $this->keys = $this->items = array();
        $this->rewind();
        return $this;
    }

    /**
     * @function add : adds an item
     * This can not accept composed keys ! Use "set" if you need that.
     * @param $item
     * @param $key
     * @param $position
     */
    public function add($items, $offset = null) {

        if (!is_array($items)) {
            $items = [ $items ];
        }

        $associative = array_keys($items) !== range(0, count($items) - 1);

        if ($offset === null) {
            // simply merge with eventual keys
            $this->items = array_merge( $this->items, $items );
        } else {
            if ($associative) {
                // manually slice
                $this->items = array_slice($this->items, 0, $offset, true) + $items + array_slice($this->items, $offset, NULL, true);
            } else {
                // simple slice
                array_splice( $this->items, $offset, 0, $items );
            }
        }

        $this->keys = array_keys($this->items);
        return $this;
    }

    public function remove_key($key) {
        Glb_Hash::remove($this->items, $key);
        $this->keys = array_keys($this->items);
        return $this;
    }

    public function remove_value($value) {
        $key = array_search($value, $this->items);
        unset($this->items[$key]);
        $this->keys = array_keys($this->items);
        return $this;
    }

    public function is_associative() {
        return array_keys($this->items) !== range(0, count($this->items) - 1);
    }

    public function is_sequential() {
        return !$this->is_associative();
    }

    public function count() {
        return count($this->items);
    }

    /**
     * Extract a single column from a collection of associative arrays / ArrayObjects / Glb_Entity
     * @param string $column : the column name to be extracted, can be a callback
     * @return array : the resulting array
     *
     * @usage :
     *      $col = new Glb_Collection([
     *          ['id' => 1, 'name' => 'syphilis'],
     *          ['id' => 2, 'name' => 'herpes'],
     *          ['id' => 3, 'name' => 'cold'],
     *          ['id' => 4, 'name' => 'pneumonia'],
     *          ['id' => 5, 'name' => 'politics'],
     *      ]);
     *      $col->extract('name') returns : ['syphilis', 'herpes', 'cold', 'pneumonia', 'politics']
     *      $col->extract(function($item, $key) { return $item['name'] . '-' . $key; }}) returns : ['syphilis-0', 'herpes-1', 'cold-2', 'pneumonia-3', 'politics-4']
     */
    public function extract($column) {
        $result = [];
        if (is_callable($column)) {
            foreach($this->items as $key => $value) {
                $result[] = $column($value, $key);
            }
        } else {
            $result = array_column($this->items, $column);
        }
        return $result;
    }

    /**
     * Combine columns to a new Array, too complicated to explain, please look at examples
     * @param $columns : the column names
     * @return array : the resulting array
     *
     * @usage :
     *      $col = new Glb_Collection([
     *          ['id' => 1, 'name' => 'syphilis', 'severity' => '4'],
     *          ['id' => 2, 'name' => 'herpes', 'severity' => '4'],
     *          ['id' => 3, 'name' => 'cold', 'severity' => '1'],
     *          ['id' => 4, 'name' => 'cold', 'severity' => '10', 'organ' => 'brain'],
     *          ['id' => 5, 'name' => 'politics', 'severity' => 'PHP_INT_MAX' ],
     *      ]);
     *
     *      $col->combine('id', 'name') returns :
     *          [1 => 'syphilis', 2 => 'herpes', 3 => 'cold', 4 => 'pneumonia', 5 => 'politics']
     *
     *      $col->combine('id', 'name', 'severity') returns :
     *          [1 => ['syphilis' => '4'] , 2 => ['herpes' => '4'], 3 => ['cancer' => '9'], 4 => ['pneumonia' => '9'], 5 => ['politics' => 'PHP_INT_MAX']]
     *
     *      $col->combine('severity', 'name') returns :
     *          ['4' => ['syphilis', 'herpes'], '1' => 'cold', '10' => 'cold', 'PHP_INT_MAX' => 'politics']
     *
     *      $col->combine('name', 'severity', 'organ') returns :
     *          ['syphilis' => [4 => null], 'herpes' => [4 => null], 'cold' => ['1' => null, '10' => 'brain']], 'politics' => ['PHP_INT_MAX' => null]]
     *
     *      $col->combine('id', function($item) { return $item['name'] . '-' . $item['severity'] }) returns :
     *          [1 => 'syphilis-4', 2 => 'herpes-4', 3 => 'cancer-9', 4 => 'pneumonia-9', 5 => 'politics-PHP_INT_MAX']
     *
     *      Capito ?
     */
    public function combine(...$columns) {

        if (func_num_args() < 2) {
            throw new Exception('combine needs at least two parameters ! Try again...');
        }

        $result = [];

        foreach($this->items as $item) {
            $item_result = $this->_combine_item($item, $columns);
            Glb_Hash::append($result, $item_result[0], $item_result[1]); //xxxxx
        }

        return $result;

    }

    private function _combine_item($item, $columns) {

        $result = [];
        $key_column = array_shift($columns);
        if (is_callable($key_column)) {
            $result[0] = $key_column($item);
        } else {
            $result[0] = Glb_Hash::get($item, $key_column);
        }

        if (count($columns) == 1) {
            if (is_callable($columns[0])) {
                $result[1] = $columns[0]($item);
            } else {
                if (Glb_Hash::exists($item, $columns[0])) {
                    $result[1] = Glb_Hash::get($item, $columns[0]);
                } else {
                    $result[1] = null;
                }
            }
        } else {

            $sub_value = $this->_combine_item($item, $columns);
            $result[0] .= '.' . $sub_value[0];
            $result[1] = $sub_value[1];

        }

        return $result;
    }

    public function get($key, $default = null) {
        return Glb_Hash::get($this->items, $key, $default);
    }

    public function set($key, $value) {
        Glb_Hash::set($this->items, $key, $value);
        $this->keys = array_keys($this->items);
        return $this;
    }
    public function append($key, $value) {
        //glb_dump($key . ' = ' . $value);
        //glb_dump($this->items);
        Glb_Hash::append($this->items, $key, $value);
        //glb_dump($this->items);
        $this->keys = array_keys($this->items);
        return $this;
    }

     /**
     * clone a Glb_Collection
     */
    public function copy() {
        return new Glb_Collection($this->items);
    }

    /**
     * Easily filter this collection to create a new collection of elements matching a criteria callback
     * @param string|callable $callback : if string, then apply the function to all values
     * @usage :
     *      $col = new Glb_Collection([
     *          ['id' => 1, 'name' => 'syphilis', 'severity' => '4'],
     *          ['id' => 2, 'name' => 'herpes', 'severity' => '4'],
     *          ['id' => 3, 'name' => 'cold', 'severity' => '1'],
     *          ['id' => 4, 'name' => 'cold', 'severity' => '10', 'organ' => 'brain'],
     *          ['id' => 5, 'name' => 'politics', 'severity' => 'PHP_INT_MAX' ],
     *      ]);
     *
     *      $col->filter(function($value, $key = false) { return ($value['id'] % 2); })
     *          returns
     */
    public function filter($callback, $flag = 0) {

        $result = new Glb_Collection();

        foreach($this->items as $key => $value) {
            // first manage string callbacks (needs to be a defined global function)
            if (is_string($callback)) {
                if (is_array($value)) {
                    foreach($value as $value_key => $value_value) {
                        $new_key = $value_key; $new_value = $value_value;
                        $accept = false;
                        if ($flag == 0) {
                            $accept = $new_value = call_user_func($callback, $new_value);
                        } elseif ($flag == ARRAY_FILTER_USE_KEY) {
                            $accept = $new_key = call_user_func($callback, $new_key);
                        } elseif ($flag == ARRAY_FILTER_USE_BOTH) {
                            $accept = $new_value = call_user_func($callback, $new_value, $new_key);
                        }
                        if ($accept) {
                            if (!$result->key_exists($key)) {
                                $result->add([$key => []]);
                            }
                            $result->append($key . '.' . $new_key, $new_value);
                        }
                    }
                } else {
                    $new_key = $key; $new_value = $value;
                    $accept = false;
                    if ($flag == 0) {
                        $accept = $new_value = call_user_func($callback, $new_value);
                    } elseif ($flag == ARRAY_FILTER_USE_KEY) {
                        $accept = $new_key = call_user_func($callback, $new_key);
                    } elseif ($flag == ARRAY_FILTER_USE_BOTH) {
                        $accept = $new_value = call_user_func($callback, $value, $new_key);
                    }
                    if ($accept) {
                        $result->set($new_key, $new_value);
                    }
                }
            } else if (is_callable($callback)) {
                $new_key = $key; $new_value = $value;
                $accept = false;
                if ($flag == 0) {
                    $accept = $callback($new_value);
                } elseif ($flag == ARRAY_FILTER_USE_KEY) {
                    $accept = $callback($new_key);
                } elseif ($flag == ARRAY_FILTER_USE_BOTH) {
                    $accept = $callback($new_value, $new_key);
                }
                if ($accept) {
                    $result->set($new_key, $new_value);
                }
            }
        }
        return $result;
    }

    public function key_exists($key) {
        return Glb_Hash::exists($this->items, $key);
    }

    public function exists($value) {
        return in_array($value, $this->items);
    }

    public function is_empty() {
        return empty($this->items);
    }

    public function one_empty() {
        foreach($this->items as $item) {
            if (empty($item)) {
                return true;
            }
        }
        return false;
    }

    public function first() {
        return $this->at(0);
    }

    public function last() {
        return $this->at(count($this->items)-1);
    }

    public function at($offset) {
        if ($this->count() <= $offset) {
            return null;
        }
        return array_values($this->items)[$offset];
    }

    /**
     * Pop the element off the end of array
     */
    public function pop() {
        $result = array_pop($this->items);
        $this->keys = array_keys($this->items);
        return $result;
    }

    /**
     * Shift an element off the beginning of array
     */
    public function shift() {
        $result = array_shift($this->items);
        $this->keys = array_keys($this->items);
        return $result;
    }

    /**
     * @function serialize
     * Returns a string representation of this object that can be used
     * to reconstruct it
     *
     * @return string
     */
    public function serialize() {
        return serialize($this->items);
    }

    /**
     * @function unserialize
     * Unserializes the passed string and rebuilds the Collection instance
     *
     * @param string $serialization The serialized collection
     * @return void
     */
    public function unserialize($serialization)
    {
        $this->__construct(unserialize($serialization));
        return $this;
    }

    public function jsonSerialize() {
        return json_encode($this->items);
    }

    public function jsonUnserialize($json) {
        $this->__construct(json_decode($json, true));
        return $this;
    }

    public function order($key, $type = 'text', $direction = 'asc') {
        $direction = strtolower($direction);
        uasort($this->items, function ($a, $b) use ($key, $type, $direction) {
            $a = (!empty($a[$key]) ? strtolower($a[$key]) : '');
            $b = (!empty($b[$key]) ? strtolower($b[$key]) : '');
            if ($a == $b) { return 0; }
            $result = 0;
            if (in_array($type, ['text', 'string', 'varchar', 'char'])) {
                $result = strcmp($a, $b);
            } else {
                if ($a == $b) {
                    $result = 0;
                } else {
                    $result = ($a < $b);
                }
            }

            if ($direction == 'desc') {
                $result = -$result;
            }
            return $result;
        });
        $this->keys = array_keys($this->items);
        return $this;
    }

}
