<?php

/*
 * Glb_Db class. Wrapper helper for $wpdb
 */
final class Glb_Data_Type extends Glb_Entity
{

    const TYPE_INTEGER = 'integer';
    const TYPE_DECIMAL = 'decimal';
    const TYPE_UUID = 'uuid';
    const TYPE_TEXT = 'text';
    const TYPE_BINARY = 'binary';
    const TYPE_JSON = 'json';
    const TYPE_SERIALIZE = 'serialize';
    const TYPE_DATE = 'date';
    const TYPE_TIME = 'time';
    const TYPE_DATETIME = 'datetime';
    const TYPE_ENUM = 'enum';
    const TYPE_SET = 'set';

    const HAS_ERROR = 1;
    const HAS_WARNING = 2;

    protected static $_map = [

        'bit' => ['name' => self::TYPE_INTEGER, 'storage' => 1],
        'bool' => ['name' => self::TYPE_INTEGER, 'storage' => 1],
        'boolean' => ['name' => self::TYPE_INTEGER, 'storage' => 1],
        'tinyint' => ['name' => self::TYPE_INTEGER, 'storage' => 8],
        'tinyinteger' => ['name' => self::TYPE_INTEGER, 'storage' => 8],
        'smallint' => ['name' => self::TYPE_INTEGER, 'storage' => 16],
        'smallinteger' => ['name' => self::TYPE_INTEGER, 'storage' => 16],
        'mediumint' => ['name' => self::TYPE_INTEGER, 'storage' => 24],
        'mediuminteger' => ['name' => self::TYPE_INTEGER, 'storage' => 24],
        'int' => ['name' => self::TYPE_INTEGER, 'storage' => 32],
        'integer' => ['name' => self::TYPE_INTEGER, 'storage' => 32],
        'bigint' => ['name' => self::TYPE_INTEGER, 'storage' => 64],
        'biginteger' => ['name' => self::TYPE_INTEGER, 'storage' => 64],

        'float' => ['name' => self::TYPE_DECIMAL, 'storage' => 32],
        'real' => ['name' => self::TYPE_DECIMAL, 'storage' => 64],
        'double' => ['name' => self::TYPE_DECIMAL, 'storage' => 64],
        'double precision' => ['name' => self::TYPE_DECIMAL, 'storage' => 64],
        'decimal' => ['name' => self::TYPE_DECIMAL],
        'numeric' => ['name' => self::TYPE_DECIMAL],

        'date' => ['name' => self::TYPE_DATE, 'min' => '1000-01-01', 'max' => '9999-12-31'],
        'datetime' => ['name' => self::TYPE_DATETIME, 'min' => '1000-01-01 00:00:00', 'max' => '9999-12-31 23:59:59'],
        'year' => ['name' => self::TYPE_INTEGER, 'min' => 1901, 'max' => 2155],
        'time' => ['name' => self::TYPE_TIME, 'min' => '-838:59:59', 'max' => '838:59:59'],
        'timestamp' => ['name' => self::TYPE_DATETIME, 'min' => '1970-01-01 00:00:01.000000', 'max' => '2038-01-19 03:14:07.999999'],

        'char' => ['name' => self::TYPE_TEXT],
        'varchar' => ['name' => self::TYPE_TEXT],
        'tinytext' => ['name' => self::TYPE_TEXT],
        'mediumtext' => ['name' => self::TYPE_TEXT],
        'longtext' => ['name' => self::TYPE_TEXT],
        'text' => ['name' => self::TYPE_TEXT],

        'binary' => ['name' => self::TYPE_BINARY],
        'varbinary' => ['name' => self::TYPE_BINARY],
        'tinyblob' => ['name' => self::TYPE_BINARY],
        'mediumblob' => ['name' => self::TYPE_BINARY],
        'longblob' => ['name' => self::TYPE_BINARY],

        'geometry' => ['name' => self::TYPE_BINARY],
        'point' => ['name' => self::TYPE_BINARY],
        'linestring' => ['name' => self::TYPE_BINARY],
        'polygon' => ['name' => self::TYPE_BINARY],
        'geometrycollection' => ['name' => self::TYPE_BINARY],
        'multilinestring' => ['name' => self::TYPE_BINARY],
        'multipoint' => ['name' => self::TYPE_BINARY],
        'multipolygon' => ['name' => self::TYPE_BINARY],

        'json' => ['name' => self::TYPE_JSON],
        'serialize' => ['name' => self::TYPE_SERIALIZE],
        'enum' => ['name' => self::TYPE_ENUM, 'allowed' => []],
        'set' => ['name' => self::TYPE_SET, 'allowed' => []],
        '~unknown' => ['name' => self::TYPE_TEXT],

        // custom types
        'email' => ['name' => self::TYPE_TEXT, 'validator' => 'filter_var', 'validator_args' => FILTER_VALIDATE_EMAIL],
        'ip' => ['name' => self::TYPE_TEXT, 'validator' => 'filter_var', 'validator_args' => FILTER_VALIDATE_IP],
        'url' => ['name' => self::TYPE_TEXT, 'validator' => 'filter_var', 'validator_args' => FILTER_VALIDATE_URL],
        'link' => ['name' => self::TYPE_TEXT, 'validator' => 'filter_var', 'validator_args' => FILTER_VALIDATE_URL],
        'folder' => ['name' => self::TYPE_TEXT, 'validator' => 'strpbrk', 'validator_args' => "\\/?%*:|\"<>"],

    ];

    /**
     * @param $name
     * @param $infos
     *      [
     *          'name' => 'my_custom_type', 'min' => 51, 'max' => 59,
     *          'validator' => function ($value) { return $value - floor($value) < 0.5; }
     *          'resolver' => function ($value) { return new Glb_Data_Type(['name' => 'my_custom_type', 'min' => 'my_min_value', 'max' => 'my_max_value']); }
     *          'convert_to' => function ($value) { return 'convert php value to external (mostly db) value'; }
     *          'convert_from' => function ($value) { return 'convert external (mostly db) value to php value'; }
     *      ]
     */
    public static function add_type($name, $infos)
    {
        self::$_map[$name] = $infos;
    }

    public static function resolve($db_definition) {

        preg_match('/([a-z]+)(?:\(([0-9,]+)\))?\s*([a-z]+)?/i', strtolower($db_definition), $matches);
        if (!array_key_exists($matches[1], self::$_map)) {
            return new Glb_Data_Type([
                'name' => $matches[1],
                'has_error' => 1,
                'errors' => ['Glb_Data_Type :: Unable to find type ' . Glb_Text::quote_label($matches[1]) . ' from definition ' . Glb_Text::quote_label($db_definition)]
            ], false);
        }
        $type = self::$_map[$matches[1]];
        $type['source'] = $db_definition;
        $type['original'] = $matches[1];

        // if resolver is callable
        if (isset($type['resolver'])) {
            if (is_callable($type['resolver'])) {
                return call_user_func_array($type['resolver'], [$db_definition]);
            } else {
                return new Glb_Data_Type([
                    'name' => $matches[1],
                    'has_error' => 1,
                    'errors' => ['Glb_Data_Type :: Type resolver is not callable ' . Glb_Text::quote_label(print_r($type['resolver'], true)) . ' from definition ' . Glb_Text::quote_label($db_definition)]
                ], false);
            }
        }

        // manage "signed" information
        if (in_array($matches[count($matches)-1], ['signed', 'unsigned'])) {
            $type['signed'] = $matches[count($matches)-1];
            unset($matches[count($matches)-1]);
        }

        // calculate min/max from storage
        if (array_key_exists('storage', $type)) {
            $type['min'] = -0.5 * pow(2, $type['storage']);
            $type['max'] = 0.5 * pow(2, $type['storage']) - 1;
        }

        // adjust min/max for unsigned
        if (isset($type['signed']) && $type['signed'] === 'unsigned' && isset($type['min']) && isset($type['max'])) {
            $type['max'] = -$type['min'] + $type['max'];
            $type['min'] = 0;
        }

        // calculate min_length / max_length
        if (in_array($matches[1], ['char', 'varchar'])) {
            $type['min_length'] = 0;
            if (!empty($matches[2])) {
                $type['max_length'] = $matches[2];
            } else {
                $type['max_length'] = 255;
            }
        }

        // calculate values for set/enum
        if (in_array($type, ['enum', 'set'])) {
            if (!empty($matches[2])) {
                $values = explode(',', $matches[2]);
                $type['values'] = [];
                foreach($values as $value) {
                    $type['values'][] = trim(str_replace('"', '', str_replace('\'', '', $value)));
                }
            }
        }

        if ($type['name'] == self::TYPE_DECIMAL && !empty($matches[2])) {
            $precision = explode(',', $matches[2]);
            // @todo calculate min/max
            $type['max_length'] = $precision[0];
            $type['dec_length'] = $precision[1];
            $type['int_length'] = $precision[0] - $precision[1];
        }

        return new Glb_Data_Type($type, false);
    }

    public function convert_from($value) {
        // specific checks
        switch ($this->name) {

            case self::TYPE_SET:
            case self::TYPE_ENUM:
                if (is_string($value)) { $value = explode(',', $value); }
                break;

            case self::TYPE_JSON:
                $value = json_decode($value, true);
                break;

            case self::TYPE_SERIALIZE:
                $value = unserialize($value);
                break;

        }
        return $value;
    }

    public function convert_to($value) {

        // specific checks
        switch ($this->name) {

            case self::TYPE_SET:
            case self::TYPE_ENUM:
                if (is_array($value)) { $value = implode(',', $value); }
                break;

            case self::TYPE_JSON:
                $value = json_encode($value);
                break;

            case self::TYPE_SERIALIZE:
                $value = unserialize($value);
                break;

            case self::TYPE_DATETIME:
                if (is_object($value) && method_exists($value, 'format')) {
                    $value = $value->format('Y-m-d H:i:s');
                } else if (is_integer($value)) {
                    $value = date('Y-m-d H:i:s', $value);
                } else if ($value == 'now') {
                    $value = date('Y-m-d H:i:s');
                }
                break;

        }
        return $value;
    }

    public function has_error() {
        return !empty($this['errors']);
    }
    public function errors() {
        if (!$this->exists('errors')) {
            return [];
        } else {
            return $this['errors'];
        }
    }

    public function has_warning() {
        return !empty($this['warnings']);
    }

    public function warnings() {
        if (!$this->exists('warnings')) {
            return [];
        } else {
            return $this['warnings'];
        }
    }
    protected function add_error($error) {
        if (!$this->exists('errors')) {
            $this->errors = [];
        }
        $this->errors[] = $error;
    }

    protected function add_warning($warning) {
        if (!$this->exists('warnings')) {
            $this->warnings = [];
        }
        $this->warnings[] = $warning;
    }

    protected function _validate_preg_match($value, $args) {
        if (is_array($args)) {  $args = $args[0]; }
        return (preg_match($args, $value) !== false);
    }

    protected function _validate_strpbrk($value, $args) {
        if (is_array($args)) {  $args = $args[0]; }
        return (strpbrk($value, $args) === false);
    }

    protected function _validate_filter_var($value, $args) {
        if (is_array($args)) {  $args = $args[0]; }
        return (filter_var($value, $args) !== false);
    }

    /**
     * Return true / error message
     * @param $value
     * @return bool|mixed
     */
    public function validate($value)
    {

        $this['errors'] = [];

        // if resolver is callable
        if (isset($this->validator)) {
            if (is_callable([$this, '_validate_' . $this->validator])) {
                return $this->{'_validate_' . $this->validator}($value, Glb_Hash::get($this, 'validator_args'));
            } else if (is_callable($this->validator)) {
                return call_user_func_array($this->validator, [$value, Glb_Hash::get($this, 'validator_args'), $this]);
            } else {
                $this->add_error(sprintf('Glb_Data_Type::%s. Validator should be callable for %s.', $this->source, Glb_Text::quote_label($this->source)));
                return false;
            }
        } else if (is_callable([$this, '_validate_' . $this->original])) {
            return $this->{'_validate_' . $this->original}($value, Glb_Hash::get($this, 'validator_args'));
        }

        // check min_length
        if ($this->exists('min_length') && strlen(strval((string)$value)) < $this->min_length) {
            $this->add_warning(sprintf('Glb_Data_Type::%s. Min length (%s) not respected for %s.', $this->source, $this->min_length, Glb_Text::quote_label(strval((string)$value))));
        }

        // check max_length
        if ($this->exists('max_length') && (strlen(strval((string)$value)) > $this->max_length)) {
            $this->add_warning(sprintf('Glb_Data_Type::%s. Max length (%s) not respected for %s.', $this->source, $this->max_length, Glb_Text::quote_label(strval((string)$value))));
        }

        // check values
        if ($this->exists('values')) {
            if (is_array($value)) {
                foreach ($value as $item) {
                    if (!in_array($item, $this->values)) {
                        $this->add_error(sprintf('Glb_Data_Type::%s. Value not allowed (%s) for %s.', $this->source, $this->values, Glb_Text::quote_label($item)));
                    }
                }
            } else {
                if (!in_array($value, $this->values)) {
                    $this->add_error(sprintf('Glb_Data_Type::%s. Value not allowed (%s) for %s.', $this->source, $this->values, Glb_Text::quote_label($value)));
                }
            }
        }

        $result = true;

        // specific checks
        switch ($this->name) {

            case self::TYPE_DECIMAL:
                // @todo : check integer/decimal parts length
                $result = filter_var($value, FILTER_VALIDATE_FLOAT);
                break;

            case self::TYPE_INTEGER:
                $result = filter_var($value, FILTER_VALIDATE_INT);
                break;

            case self::TYPE_TEXT:
                $result = ($value == wp_check_invalid_utf8($value));
                break;

            case self::TYPE_ENUM:
                $result = !is_array($value);
                break;

            case self::TYPE_JSON:
                $result = Glb_Text::is_json($value);
                break;

            case self::TYPE_DATETIME:
                $result = Glb_Text::is_date($value, 'Y-m-d H:i:s');
                break;

            case self::TYPE_DATE:
                $result = Glb_Text::is_date($value, 'Y-m-d');
                break;

            case self::TYPE_TIME:
                $result = Glb_Text::is_date($value, 'H-i-s');
                break;

            case self::TYPE_SERIALIZE:
                $result = Glb_Text::is_serialized($value);
                break;

        }
        if ($result === false) {

            $this->add_error(sprintf('Glb_Data_Type::%s. Invalid value %s.', $this->source, Glb_Text::quote_label($value)));

        } else {
            // check min
            if ($this->exists('min') && $value < (float)$this->min) {
                $this->add_warning(sprintf('Glb_Data_Type::%s. Min value (%s) not respected for %s.', $this->source, $this->min, Glb_Text::quote_label($value)));
            }

            // check max
            if ($this->exists('max') && $value > (float)$this->max) {
                $this->add_warning(sprintf('Glb_Data_Type::%s. Max value (%s) not respected for %s.', $this->source, $this->max, Glb_Text::quote_label($value)));
            }
        }

       return !$this->has_error() && !$this->has_warning();
    }

}
