<?php

class Glb_Request
{

    /**
     * @var array $_detectors The detectors that can be applied in Glb_Request::is() function
     */
    protected $_detectors = [
        'get'       => ['env' => 'REQUEST_METHOD', 'value' => 'GET'],
        'post'      => ['env' => 'REQUEST_METHOD', 'value' => 'POST'],
        'put'       => ['env' => 'REQUEST_METHOD', 'value' => 'PUT'],
        'patch'     => ['env' => 'REQUEST_METHOD', 'value' => 'PATCH'],
        'delete'    => ['env' => 'REQUEST_METHOD', 'value' => 'DELETE'],
        'head'      => ['env' => 'REQUEST_METHOD', 'value' => 'HEAD'],
        'options'   => ['env' => 'REQUEST_METHOD', 'value' => 'OPTIONS'],
        'ssl'       => ['env' => 'HTTPS', 'options' => [1, 'on']],
        'ajax'      => ['env' => 'HTTP_X_REQUESTED_WITH', 'value' => 'XMLHttpRequest'],
        'flash'     => ['env' => 'HTTP_USER_AGENT', 'pattern' => '/^(Shockwave|Adobe) Flash/'],
        'json'      => ['accept' => ['application/json'], 'param' => '_ext', 'value' => 'json'],
        'xml'       => ['accept' => ['application/xml', 'text/xml'], 'param' => '_ext', 'value' => 'xml'],
    ];


    /**
     * @var array $_actions_config The config used for actions / failures actions
     */
    protected $_actions_config = [
        'abbrevs' => [
            'status:set'            => [ 'type' => 'status', 'value' => '{{status_code}}' ],                    // add notice according to status_code
            'notice:status'         => [ 'type' => 'notice', 'value' => '{{status_text}}' ],                    // add notice according to status_code
            'notice:invalid_field'  => [ 'type' => 'notice', 'value' => 'Invalid value for « {{field}} »' ],    // add notice according to status_code
            'redirect:referer'      => [ 'type' => 'redirect', 'value' => '{{referer_url}}' ],                  // redirect to referer
            'redirect:home'         => [ 'type' => 'redirect', 'value' => '{{home_url}}' ],                     // redirect home
            'redirect:dashboard'    => [ 'type' => 'redirect', 'value' => '{{dashboard_url}}' ],                // redirect home
            'log:status'            => [ 'type' => 'log', 'value' => '{{status_text}}' ],                       // log in file using status code for severity
            'log:invalid_field'     => [ 'type' => 'log', 'value' => 'Invalid value for « {{field}} »' ],       // log in file using status code for severity
            'db_log:status'         => [ 'type' => 'db_log', 'value' => '{{status_text}}' ],                    // log in database using status code for severity
            'db_log:invalid_field'  => [ 'type' => 'db_log', 'value' => 'Invalid value for « {{field}} »' ],    // log in database invalid field
        ],
        'allowed'                   => [ 'status', 'notice', 'redirect', 'log', 'db_log' ], // all allowed action types
        'default'                   => [ 'status:set', 'notice:status', 'log:status', 'db_log:status', 'redirect:referer' ]
    ];

    public static function instance() {
        static $instance = null;
        if ($instance === null) {
            $instance = new Glb_Request();
        }
        return $instance;
    }

    public function add_detector($detector_key, $detector_value) {
        $this->_detectors[$detector_key] = $detector_value;
    }

    public function __construct() {
        $this->add_detector('heartbeat', function() {
            return (is_admin() && defined( 'DOING_AJAX' ) && DOING_AJAX && !empty($_POST['action']) && $_POST['action'] == 'heartbeat');
        });
    }

    /**
     * Get Gloubi Boulga status message according to the code
     * @param $error_code
     * @return mixed|null
     */
    private function get_status_message($error_code) {

        $http_status_messages = [
            200 => __glb('OK'),
            400 => __glb('Bad request, buddy :-('),
            401 => __glb('You tried to enter the house without the keys, that\'s bad.'),
            402 => __glb('Payment Required. Don\'t know why...'),
            403 => __glb('Wooww... you can\'t do that !'),
            404 => __glb('Not found. Maybe on vacation.'),
            500 => __glb('Terrible internal error, can\'t say more but it\'s very serious !'),
        ];

        if (array_key_exists($error_code, $http_status_messages)) {
            return $http_status_messages[$error_code];
        } else {
            return null;
        }
    }

    /*
    * get value from environment variable
    */
    public function get_env($key, $default = null) {
        $key = strtoupper($key);
        if (getenv($key) === false) {
            return $default;
        } else {
            return getenv($key);
        }
    }

    /*
     * get $_ENV http_$key value
     */
    public function get_header($key, $default = null) {
        $header = getEnv(strtoupper('http_' . $key));
        if ($header === false) {
            return $default;
        } else {
            return $header;
        }
    }

    /*
    * get HTTP_ACCEPT value from $_ENV variable
    */
    public function get_accept($key, $default= null) {
        $acceptHeaders = explode(',', getEnv('HTTP_ACCEPT'));
        if (in_array($key, $acceptHeaders)) {
            return true;
        }
        return false;
    }

    public function is($types) {
        Glb_Array::ensure($types);
        $result = false;

        foreach($types as $type) {

            $detect = $this->_detectors[$type];

            if (is_callable($detect)) {
                if ($detect($type)) {
                    return true;
                }
            } else if (isset($detect['env'])) {
                if (isset($detect['value'])) {
                    if (static::get_env($detect['env']) == $detect['value']) {
                        return true;
                    }
                }
                if (isset($detect['pattern'])) {
                    if ((bool)preg_match($detect['pattern'], static::get_env($detect['env']))) {
                        return true;
                    }
                }
                if (isset($detect['options'])) {
                    $pattern = '/' . implode('|', $detect['options']) . '/i';
                    if ((bool)preg_match($pattern, static::get_env($detect['env']))) {
                        return true;
                    }
                }
            } else if (isset($detect['header']) && $this->get_header($detect['header'])) {
                return true;
            } else if (isset($detect['accept']) && $this->get_accept($detect['header'])) {
                return true;
            }
            /*if (isset($detect['param']) && $this->_paramDetector($detect)) {
                return true;
            }*/
        }
    }

    public function is_from($types) {
        Glb_Array::ensure($types);
        foreach($types as $type) {
            if ($type == 'admin') {
                return defined('WP_ADMIN');
            } else if ($type == 'public') {
                return !defined('WP_ADMIN');
            }
        }
    }

    public function build_json_response($status, $status_text = null, $data = null, $actions = null) {
        $actions = $this->_normalize_actions($actions);
        foreach($actions as $index => $action) {
            $actions[$index]['value'] = $this->_complete_action($action['value'], 0);
        }
        return [
            'status' => $status,
            'status_desc' => ($status_text === null ? $this->get_status_message($status) : $status_text),
            'data' => $data,
            'actions' => $actions
        ];
    }

    /*private function build_send_json_response($status, $status_text, $data = null, $actions = null, $exit = false) {
        $this->_send_response($this->build_json_response($status, $status_text, $data, $actions));
        if ($exit) { exit(); }
    }*/

    /*private function _send_response($response) {
        if ($response['status'] != 200) {
            $protocol = (isset($_SERVER['SERVER_PROTOCOL']) ? $_SERVER['SERVER_PROTOCOL'] : 'HTTP/1.0');
            header($protocol . ' ' . $response['status'] . ' ' . $response['status_desc']);
            status_header($response['status'], $response['status_desc']);
        }
        echo wp_json_encode($response);
    }*/

    /**
     * @param array $header : can be formed like that ['status' => 200, 'status_text' => 'OK',
     *                                                  'code' =>  'Content-type', 'text' => 'application/zip']
     * @usage
     *
     *      Glb_Request::instance()->set_response_header(['status' => 200, 'status_text' => 'OK',
     *                                                  'code' => 'Content-type', 'text' => 'application/zip']);
     *          Will apply : header('Content-type: application/zip'); status_header(200, 'OK');

     *      Glb_Request::instance()->set_response_header(['code' => 'Content-type', 'text' => 'application/zip']);
     *          Will only apply : header('Content-type: application/zip');
     *
     *      Glb_Request::instance()->set_response_header(['status' => 200]);
     *          Will only apply :  status_header(200);
     *
     */
    public function set_response_header($header) {
        if (array_key_exists('status', $header)) {
            $status_desc = (!array_key_exists('status_text', $header) ? $this->get_status_message($header['status']) : $header['status_text']);
            //$protocol = (isset($_SERVER['SERVER_PROTOCOL']) ? $_SERVER['SERVER_PROTOCOL'] : 'HTTP/1.0');
            //header($protocol . ' ' . $header['status'] . ' ' . $status_desc);
            status_header($header['status'], $status_desc );
        }
        if (array_key_exists('code', $header)) {
            header($header['code'] . ': ' . $header['text']);
        }
    }

    public function get_post($var = null, $default = null, $raw = false) {

        $data = $this->_filter_post($raw);
        if ($var === null) {
            return $data;
        }
        if (array_key_exists($var, $data)) {
            return $data[$var];
        }
        return $default;
    }

    /*public static function get_post_equals($var, $value, $strict = false) {
        if (array_key_exists($var, $_POST)) {
            if ($strict) {
                return ($_POST[$var] === $value);
            } else {
                return ($_POST[$var] == $value);
            }
        }
        return false;
    }*/

    /*public function encode_post_names($name, $replacements = ['.' => '~~~']) {
        $result = $name;
        foreach($replacements as $replacement_key => $replacement_value) {
            $result = str_replace($replacement_key, $replacement_value, $result);
        }
        return $result;
    }

    public function decode_post_name($name, $replacements = ['.' => '~~~']) {
        $result = $name;
        foreach($replacements as $replacement_key => $replacement_value) {
            $result = str_replace($replacement_key, $replacement_value, $result);
        }
        return $result;
    }*/

    private function _filter_post($raw = false) {
        if ($raw) {
            $query = file_get_contents('php://input');
            $data = [];
            parse_str($query, $data);
        } else {
            $data = $_POST;
        }
        if ($this->is('ajax')) {
            $data = wp_unslash($data);
        }
        return $data;
    }

    private function _filter_get() {
        return $_GET;
    }

    public function get_post_or_get($var = null, $default = null, $raw = false) {

        if ($var == null) {
            return array_merge($this->_filter_get(), $this->_filter_post($raw));
        }

        $data = $this->_filter_post();
        if (array_key_exists($var, $data)) {
            return $data[$var];
        }
        $data = $this->_filter_get();
        if (array_key_exists($var, $data)) {
            return $data[$var];
        }
        return $default;
    }

    public function get_get($var = null, $default = null) {

        $data = $this->_filter_get();
        if ($var === null) {
            return $data;
        }
        if (array_key_exists($var, $data)) {
            return $data[$var];
        }
        return $default;
    }

    /*public function get_session($var = null, $default = null) {
        return Glb_Session::instance()->get($var, $default);
    }

    public function set_session($var, $val) {
        Glb_Session::instance()->set($var, $val);
    }*/

    /*
     * @usage :
     *      for URL http://my-domain.com/wp-admin/admin.php?page=glb_admin_page&tab=settings
     *      get_url(true) returns   http://my-domain.com/wp-admin/admin.php
     *      get_url(false) returns  http://my-domain.com/wp-admin/admin.php?page=glb_admin_page&tab=settings
     */
    public function get_url($remove_args = true) {
        if ($remove_args) {
            $parsed = parse_url($_SERVER["REQUEST_URI"], PHP_URL_PATH);
            return (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://{$_SERVER['HTTP_HOST']}$parsed";
        } else {
            return (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://{$_SERVER['HTTP_HOST']}{$_SERVER['REQUEST_URI']}";
        }

    }

    /*
     * @usage :
     *      for URL http://my-domain.com/wp-admin/admin.php?page=glb_admin_page&tab=settings
     *      get_uri(true) returns   /wp-admin/admin.php
     *      get_uri(false) returns  /wp-admin/admin.php?page=glb_admin_page&tab=settings
     */
    public function get_uri($remove_args = true) {
        if ($remove_args) {
            $parsed = parse_url($_SERVER["REQUEST_URI"], PHP_URL_PATH);
            return $parsed;
        } else {
            return $_SERVER['REQUEST_URI'];
        }
    }

    public function get_referer($remove_args = null) {
        $url = wp_get_referer();
        return $this->_remove_url_args($url, $remove_args);
    }

    /*
     * @function check_url : test conditions for url
     * @param array $conditions : a list of conditions
     *      $conditions = [
     *              'uri_contains' => ['wp_admin', 'glb_core_admin'],
     *              'uri_equals' => ['wp-admin/admin.php?page=glb_admin_page&tab=settings', 'wp-admin/admin.php?page=glb_admin_page'],
     *              'uri_args_equals' => ['page=glb_admin_page&tab=settings'],
     *              'uri_args_contains' => ['page=glb_admin_page', 'tab=settings'],
     * ]
     */
    public function check_url($conditions = ['key_word' => ['possible_values']]) {

        $uri = $this->get_uri(false);
        $args = str_replace($this->get_uri(true), '', $uri);

        $result = false;

        if (!empty($conditions['uri_contains'])) {
            Glb_Array::ensure($conditions['uri_contains']);
            foreach($conditions['uri_contains'] as $uri_contains) {
                if (strpos($uri, $uri_contains) !== false) {
                    $result = true;
                    break;
                }
            }
            if (!$result) {
                return false;
            }
        }

        if (!empty($conditions['uri_equals'])) {
            Glb_Array::ensure($conditions['uri_equals']);
            foreach($conditions['uri_equals'] as $uri_equals) {
                if ($uri == $uri_equals) {
                    $result = true;
                    break;
                }
            }
            if (!$result) {
                return false;
            }
        }

        if (!empty($conditions['uri_args_contains'])) {
            Glb_Array::ensure($conditions['uri_args_contains']);
            foreach($conditions['uri_args_contains'] as $uri_args_contains) {
                if (strpos($args, $uri_args_contains) !== false) {
                    $result = true;
                    break;
                }
            }
            if (!$result) {
                return false;
            }
        }

        if (!empty($conditions['uri_args_equals'])) {
            Glb_Array::ensure($conditions['uri_args_equals']);
            foreach($conditions['uri_args_equals'] as $uri_args_equals) {
                if ($args == $uri_args_equals) {
                    $result = true;
                    break;
                }
            }
            if (!$result) {
                return false;
            }
        }

        return $result;
    }

    protected function _remove_url_args($url, $args) {
        if (empty($args)) {  return $url; }

        foreach ($args as $arg) {
            $url = remove_query_arg($arg, $url);
        }
        return $url;
    }

    /*public static function is_ajax() {
        //return (defined('DOING_AJAX') && DOING_AJAX);
        return wp_doing_ajax();
    }*/

    /*public static function is_post() {
        return !empty($_POST);
    }*/

    /*public static function is_admin_ajax() {
        return wp_doing_ajax() && defined('WP_ADMIN');
    }*/


    /**
     * @param $status_code
     * @param array|null|false $actions If null, then it will apply default actions. If false or [], no actions will be applied.
     * @param bool $exit
     */
    public function set_error($status_code, $actions = null, $exit = false, $options = []) {
        if ($this->is('ajax')) {
            $this->set_json_error($status_code, $actions, $exit, $options);
        } else {
            $this->set_html_error($status_code, $actions, $exit, $options);
        }
    }

    public function set_json_error($status_code, $actions = null, $exit = true, $options = []) {
        if ($actions === null) {
            if (array_key_exists('default_' . $status_code, $this->_actions_config)) {
                $actions = $this->_actions_config['default_' . $status_code];
            } else {
                $actions = $this->_actions_config['default'];
            }
        }
        //$this->set_response_header([ 'status' => $status_code, 'status_text' => $this->get_status_message($status_code) ]);
        $actions = $this->_normalize_actions($actions);
        $this->_process_error('json', $status_code, $actions);
        $this->_process_actions($actions, $status_code, Glb_Hash::get($options, 'field'));
        $json = $this->build_json_response($status_code, null, null, $actions);
        echo wp_json_encode($json);
        if ($exit) { exit(); }
    }

    public function set_html_error($status_code, $actions, $exit = false, $options = []) {
        if ($actions === null) {
            $actions = $this->_actions_config['default'];
        }
        //$this->set_response_header([ 'status' => $status_code, 'status_text' => $this->get_status_message($status_code) ]);
        $actions = $this->_normalize_actions($actions);
        $this->_process_error('html', $status_code, $actions);
        $this->_process_actions($actions, $status_code, Glb_Hash::get($options, 'field'));
        if ($exit) { exit(); }
    }

    private function _process_error($type, $status_code, $actions) {
        //Glb_Log::notice("$type ERROR $status_code : " . $_SERVER['REQUEST_URI'] . ' ');
        //Glb_Log::notice('POST : ' . print_r($_POST, true));
        //Glb_Log::notice('GET : ' . print_r($_GET, true));
        //Glb_Log::notice("ACTIONS : " . print_r($actions, true));
    }


    /**
     * Analyse set_error or set_response passed $actions to convert abbreviations to real
     * well-formatted actions. For internal use only.
     *
     * @param string $action
     * @return array The well formatted expected actions
     */
    private function _normalize_actions($actions) {

        // no action, this is a basic developer right
        if (empty($actions)) {
            return $actions;
        }

        // first simpliest case : $actions is a string
        // example : something like 'redirect:referer'
        if (is_string($actions)) {

            if (array_key_exists($actions, $this->_actions_config['abbrevs'])) {
                return [ $this->_actions_config['abbrevs'][$actions] ];
            } else {
                // developer should be sure of its actions !
                throw new Exception("Glb_Request::_normalize_actions: unknown action « " . $actions . " »");
            }

        // then if $actions is an array
        } else if (is_array($actions)) {

            // first check if we have a keyword in key list, so, it would mean that we have a single action
            // example : something like ['redirect' => home_url()]
            if (Glb_Hash::first($actions, ['type', 'value']) !== null) {
                return [ $actions ];
            }

            // then we can expect to deal with an array of actions
            // example : something like ['notice' => 'You Are My Hero !', 'redirect' => get_dashboard_url()]
            $result = [];
            foreach($actions as $action) {
                if (is_string($action)) {
                    if (array_key_exists($action, $this->_actions_config['abbrevs'])) {
                        $result[] = $this->_actions_config['abbrevs'][$action];
                    } else {
                        // developer should be sure of its actions !
                        throw new Exception("Glb_Request::_normalize_actions: unknown action « " . $action . " »");
                    }
                } else if (in_array($action['type'], $this->_actions_config['allowed'])) {
                    $result[] = $action;
                }
            }
            return $result;
        }
        return $actions;
    }

    private function _complete_action($action, $status_code, $field = null) {

        if (!is_string($action)) {
            return $action;
        }

        $variables = Glb_Text::between_all($action, '{{', '}}');
        if (empty($variables)) { return $action; }

        foreach($variables as $variable) {
            $variable_value = '';
            if ($variable == 'status_text') {
                $variable_value = $this->get_status_message($status_code);
            } else if ($variable == 'status_code') {
                $variable_value = $status_code;
            } else if ($variable == 'field') {
                $variable_value = empty($field) ? '?' : $field;
            } else if ($variable == 'referer_url') {
                $variable_value = $this->get_referer();
            } else if ($variable == 'home_url') {
                $variable_value = home_url();
            } else if ($variable == 'dashboard_url') {
                $variable_value = get_dashboard_url();
            } else {
                $variable_value = '{{' . $variable . '}}';
            }
            $action = str_replace('{{' . $variable . '}}', $variable_value, $action);
        }
        return $action;
    }


    /**
     * For internal use only. Here, $actions are expected to be normalized
     * Something like that : [ [ 'notice' => 'You Are My Hero !' ], [ 'redirect' => 'http://mysite/' ] ]
     * @param array $actions
     * @throws Exception
     */
    private function _process_actions(&$actions, $status_code, $field) {

        $i = 0;

        foreach($actions as $action_index => $action) {

            $i++;
            if (!in_array($action['type'], $this->_actions_config['allowed'])) {
                throw new Exception("Glb_Request::_process_actions: unknown action type in « " . print_r($action, true) . " »");
            }

            $action['value'] = $actions[$action_index]['value'] = $this->_complete_action($action['value'], $status_code, $field);

            // processed first, but need to come last in case there is multiple actions
            if ($action['type'] == 'redirect') {

                // redirect action should come last
                if ($i < count($actions)) {
                    // just a warning
                    Glb_Log::warning("Glb_Request::_process_actions: action « redirect » should come last in « " . print_r($actions, true) . " »");
                }

                // here, we only process html redirects, because ajax redirects are processed in javascript
                if (!$this->is('ajax')) {
                    wp_redirect($action['value'], 302);
                }

            } else if ($action['type'] == 'notice') {


                $severity = 'info';
                if (array_key_exists('severity', $action)) {
                    $severity = $action['severity'];
                } else if (!empty($status_code) && $status_code != 200) {
                    $severity = 'error';
                }
                Glb_Notices::instance()->add($action['value'], $severity, Glb_Hash::get($action, 'options', []));

            } else if ($action['type'] == 'log') {

                $severity = 'info';
                if (array_key_exists('severity', $action)) {
                    $severity = $action['severity'];
                } else if (!empty($status_code) && $status_code != 200) {
                    $severity = 'error';
                }

                Glb_Log::{$severity}($action['value'] . ' from ' . $this->get_referer(), $this->get_get(), $this->get_post());

            } else if ($action['type'] == 'db_log') {

                $severity = 'info';
                if (array_key_exists('severity', $action)) {
                    $severity = $action['severity'];
                } else if (!empty($status_code) && $status_code != 200) {
                    $severity = 'error';
                }
                Glb_Db_Log::instance()->log('request_' . $severity, 'engine', [$action['value'], Glb_Hash::get($action, 'args')], null, $severity);

            } else if ($action['type'] == 'status') {

                $this->set_response_header(['status' => $action['value'], 'status_text' => $this->get_status_message($action['value'])]);

            }

        }

    }


    /*
     * @function check_permissions : check capabilities and exit if check failed
     * @param string $exit : exit if check failed
     * @param array $capabilities_conditions :
     */
    public function check_permissions($capabilities_conditions) {
        if ($this->check_conditions($capabilities_conditions)) {
            return true;
        }
        $this->set_error(403, ['redirect' => $this->get_referer()], true);
    }


    /*
     * @function get_post : get the current post
     * @return WP_Post | null : post or nothing if not a post
     */
    public function get_wp_post() {
        return get_post();
    }


    /*
     * @function get_wp_post_type : get the current post type
     * @return (string|false) Post type on success, false on failure.
     * strings can be post | page | attachment | revision | nav_menu_item...
     */
    public function get_wp_post_type() {
        return get_post_type();
    }

    /*
     * @function check_condition_item
     * @param string $type : condition type, eg IS_FROM, WITH_CAPS, FOR_PAGES, WITH_SHORTCODES
     * @param mixed $value : the value to be tester
     * @return true if condition is valid, false otherwise
     */
    private function check_condition_item($type, $value) {

        //__dump('check_condition_item : ' . $type . ' = ' . $value);
        if ($type == 'WITH_CAPS') {

            return current_user_can($value);

        } else if ($type == 'WITHOUT_CAPS') {

            return !current_user_can($value);

        } else if ($type == 'IS_FROM') {

            //__dump('is from : ' . $value . ' : ' . $this->is_from($value));
            return $this->is_from($value);

        } else if ($type == 'FOR_PAGES' && Glb_Text::starts_with($value, '[') && Glb_Text::ends_with($value, ']')) {
            //__dump('check_condition_item shortcode ' . $value);

            // shortcodes only for public
            if ($this->is_from('admin')) {
                //__dump('return null');
                return null;
            }

            $post_type = $this->get_wp_post_type();
            $post_object = $this->get_wp_post();
            // don't block if no post type (admin page for example)
            //__dump('post_type');
            //__dump($post_object);
            //__dump(Glb_String::get_between($value, '[', ']'));
            //__dump($post_object);
            if (empty($post_type) || empty($post_object)) { return true; }
            return (is_a( $post_object, 'WP_Post') && has_shortcode( $post_object->post_content, Glb_Text::between($value, '[', ']')) );

        } else if ($type == 'FOR_PAGES') {

            //__dump('return null');
            // pages only for admin
            if ($this->is_from('public')) {
                return null;
            }

            $getPage = isset($_GET['page']) ? $_GET['page'] : '';
            if (!empty($getPage) && $getPage == $value) {
                return true;
            }

        }
        return false;
    }

    /*
     * @function check_conditions
     * @param string | callable $conditions
     * check conditions before any action (load js, load css, load bud...)
     * $conditions string can be formatted this way :
     *      $conditions = 'isfrom admin withcaps manage_options or glb_core_admin and glb_core_access'
     */
    public function check_conditions($conditions) {

        //__dump('check conditions');
        //__dump($conditions);
        $render = [];
        //$conditions = strtolower($conditions);
        if (empty($conditions)) { return false; }
        $keywords = ['IS_FROM', 'WITH_CAPS', 'WITHOUT_CAPS', 'FOR_PAGES'];

        // find keywords
        $found_keywords = [];
        foreach($keywords as $keyword) {
            if (preg_match('/\\b' . $keyword . '\\b/', $conditions)){
                $found_keywords[] = $keyword;
            }
        }

        if (count($found_keywords) == 0) { return false; }

        // find keywords and content
        $base_pattern = '(' . join('|', $keywords) . ')(.*)';
        $pattern = '';
        foreach($found_keywords as $keyword) {
            $pattern .= $base_pattern;
        }
        $pattern = '/' . $pattern . '/';

        preg_match($pattern, trim($conditions), $matches);
        if (!in_array('IS_FROM', $matches)) {
            $render = ['admin', 'public'];
        }

        for ($i = 1; $i < count($matches); $i+=2) {

            $content = trim($matches[$i+1]);
            $built_content = [];
            //var_dump($i . ' ' . $content);
            if (preg_match('/\b(AND|OR)/', $content)) {
                $built_content = explode(' AND ', $content);
                foreach ($built_content as $item_key => $item) {
                    $built_content[$item_key] = explode(' OR ', $item);
                }
            } else {
                // if not "nad/or", then consider this is "and"
                $built_content = explode(' ', $content);
            }

            //var_dump($built_content);

            // parse content
            // first explode with and because of precedence
            $accepted = false;

            foreach($built_content as $and) {
                //var_dump('testing ' . print_r($and, true));
                $or_accepted = false;
                if (is_array($and)) {
                    $and = array_map('trim', $and);
                    foreach ($and as $or) {
                        if (!empty($this->check_condition_item($matches[$i], $or))) {
                            if ($matches[$i] == 'IS_FROM') { $render = $and; }
                            $or_accepted = true;
                            break;
                        }
                        /*if ($matches[$i] == 'WITH_CAPS') {
                            if (current_user_can($or)) {
                                $or_accepted = true;
                                break;
                            }
                        } else if ($matches[$i] == 'IS_FROM') {
                            if ($this->is_from($or)) {
                                $render = $and;
                                $or_accepted = true;
                                break;
                            }
                        } else if ($matches[$i] == 'WITH_SHORTCODES') {
                        } else if ($matches[$i] == 'FOR_PAGES') {

                        }*/
                    }
                } else {
                    $and = trim($and);
                    if (!empty($this->check_condition_item($matches[$i], $and))) {
                        if ($matches[$i] == 'IS_FROM') { $render[] = $and; }
                        $or_accepted = true;
                        break;
                    }

                    /*if ($matches[$i] == 'WITH_CAPS') {
                        if (current_user_can($and)) {
                            $or_accepted = true;
                            break;
                        }
                    } else if ($matches[$i] == 'IS_FROM') {
                        if ($this->is_from($and)) {
                            $render[] = $and;
                            $or_accepted = true;
                            break;
                        }
                    } else if ($matches[$i] == 'WITH_SHORTCODES') {
                    } else if ($matches[$i] == 'FOR_PAGES') {

                    }*/
                }

                if (!$or_accepted) {
                    //__dump('refused');
                    return false;
                }
            }
        }

        //__dump('accepted ');
        return $render;

    }

    /**
     * @function create_nonce : creates a cryptographic token tied to a specific action, user, user session, and window of time.
     * @param (string) $action : scalar value to add context to the nonce. Action must be set, and must be as specific as possible.
     * @return string : the token to add to the url
     */
    public function create_nonce($action) {
        return wp_create_nonce($action);
    }

    /**
     * @function update_url_nonce : update an URL with new nonce
     * @param string $url : URL to modify
     * @param string $action : action relative to the nonce
     * @param string (optional) $parameter_name : url parameter name. Default = 'glb_nonce' is used
     * @return string : the full updated URL
     */
    public function nonce_that($url, $action, $parameter_name = 'glb_nonce') {

        /*if ($action == null) {
            $bt = debug_backtrace();
            if (!empty($bt[0]['file'])) {
                $action = $bt[0]['file'];
                Glb_Log::notice($action);
            } else {
                $action = 'glb-action';
            }
        }*/
        //$parsed = parse_url($url);
        //if (empty($parsed['query'])) { $parsed['query'] = ''; }
        //parse_str($parsed['query'], $query);
        //$query['glb_nonce'] = $this->create_nonce($action);
        //$parsed['query'] = http_build_query($query);
        //return $this->unparse_url($parsed);
        add_query_arg($parameter_name, $this->create_nonce($action), $url);
    }

   /*private static function unparse_url(array $parsed) {
        $get = function ($key) use ($parsed) {
            return isset($parsed[$key]) ? $parsed[$key] : null;
        };

        $pass      = $get('pass');
        $user      = $get('user');
        $userinfo  = $pass !== null ? "$user:$pass" : $user;
        $port      = $get('port');
        $scheme    = $get('scheme');
        $query     = $get('query');
        $fragment  = $get('fragment');
        $authority =
            ($userinfo !== null ? "$userinfo@" : '') .
            $get('host') .
            ($port ? ":$port" : '');

        return
            (strlen($scheme) ? "$scheme:" : '') .
            (strlen($authority) ? "//$authority" : '') .
            $get('path') .
            (strlen($query) ? "?$query" : '') .
            (strlen($fragment) ? "#$fragment" : '');
    }*/


    /**
     * @function check_nonce : verify that correct nonce was used.
     * @param string $nonce : nonce that was used in the form to verify.
     * @param string $action : give context to what is taking place and be the same when nonce was created.
     * @param string (optional) $nonce_key : give context to what is taking place. Must be the same when nonce was created.
     * @return void|bool. If check nonce failed then exit the request invoking 'gloubi-security-failure' template
     * @todo 'gloubi-security-failure' template
     */
    public function check_nonce($action = null, $nonce_key = 'glb_nonce') {
        $nonce = $this->get_post_or_get($nonce_key);

        if (wp_verify_nonce($nonce, $action) !== 1) {
            Glb_Log::notice('isajax');
            Glb_Log::notice($this->is('ajax'));
            if ($this->is('ajax')) {
                $this->set_json_error('403', 'redirect:referer', true);
            }
            Glb_Html::add('glb_security_failure', 'core:elements/glb-core-security-failure');
            die(Glb_Html::get('glb_security_failure')->html());
        }
        return true;
    }


    /**
     * @function validate_input : check validity for input value
     * @param string $data_type : data type expected (text, string, integer, float, decimal, ip, ...)
     * @param mixed $value : input value set from outside the code (html, api, ...)
     * @param mixed $log_details : in case, validation failed, the action will be logged for security,
     *              can be the name of a setting for example.
     * @param array (option) $allowed_values : if $value must be one of $allowed_values
     * @return bool : true if everything is OK, false if NOK
     */
    public function validate_input($data_type, $value, $allowed_values = []) {
        $result = false;
        if (in_array($data_type, ['id'])) {
            $result = ($value == '~new' || (filter_var($value, FILTER_VALIDATE_INT) !== false));
        } else if (in_array($data_type, ['text', 'string'])) {
            $result = ($value == wp_check_invalid_utf8($value));
        } elseif (in_array($data_type, ['integer', 'int'])) {
            $result = (filter_var($value, FILTER_VALIDATE_INT) !== false);
        } elseif (in_array($data_type, ['decimal', 'float', 'real'])) {
            $result = (filter_var($value, FILTER_VALIDATE_FLOAT) !== false);
        } elseif (in_array($data_type, ['email'])) {
            $result = (filter_var($value, FILTER_VALIDATE_EMAIL) !== false);
        } elseif (in_array($data_type, ['boolean', 'bool'])) {
            $result = (filter_var($value, FILTER_VALIDATE_BOOLEAN, FILTER_NULL_ON_FAILURE) !== null);
        } elseif (in_array($data_type, ['ip'])) {
            $result = (filter_var($value, FILTER_VALIDATE_IP) !== false);
        } elseif (in_array($data_type, ['url', 'link'])) {
            $result = (filter_var($value, FILTER_VALIDATE_URL) !== false);
        }

        if ($result && !empty($allowed_values)) {
            $result = in_array($value, $allowed_values);
        }

        return $result;
    }


    public function check_input($data_type, $data, $field_name, $allowed_values = [],
                            $failure_actions = ['status:set', 'notice:status', 'db_log:invalid_field', 'redirect:referer'],
                            $exit = true) {

        if (is_array($data) || is_object($data)) {
            $value = Glb_Hash::get($data, $field_name);
        } else {
            $value = $data;
        }

        if ($value === null || !$this->validate_input($data_type, $value, $allowed_values)) {
            $this->set_error(400, $failure_actions, $exit, ['field' => $field_name]);
        }
        return true;
    }

    public function check_not_empty($data, $field_name = null,
                                    $failure_actions = ['status:set', 'notice:status', 'db_log:invalid_field', 'redirect:referer'],
                                    $exit = true) {

        $result = true;
        if (is_array($field_name)) {
            foreach ($field_name as $name) {
                if (!$this->check_not_empty($data, $name, $failure_actions, $exit)) {
                    $result = false;
                }
            }
            return $result;
        }

        if (is_array($data) || is_object($data)) {
            $value = Glb_Hash::get($data, $field_name);
        } else {
            $value = $data;
        }
        if (empty($value)) {
            $result = false;
            $this->set_error( 400 , $failure_actions, $exit, ['field' => $field_name] );
        }
        return $result;
    }

    public function check_not_void($data, $field_name = null,
                                   $failure_actions = ['status:set', 'notice:status', 'db_log:invalid_field', 'redirect:referer'],
                                   $exit = true) {

        $result = true;
        if (is_array($field_name)) {
            foreach ($field_name as $name) {
                if (!$this->check_not_void($data, $name, $failure_actions, $exit)) {
                    $result = false;
                }
            }
            return $result;
        }

        if (is_array($data) || is_object($data)) {
            $value = Glb_Hash::get($data, $field_name);
        } else {
            $value = $data;
        }
        if ($value === null || strlen($value) == '') {
            $result = false;
            $this->set_error( 400 , $failure_actions, $exit, ['field' => $field_name] );
        }
        return $result;
    }

    // @todo : refactor that !
    public function check_exists($data, $field_name = null,
                                   $failure_actions = ['status:set', 'notice:status', 'db_log:invalid_field', 'redirect:referer'],
                                   $exit = true) {

        $result = true;
        if (is_array($field_name)) {
            foreach ($field_name as $name) {
                if (!$this->check_exists($data, $name, $failure_actions, $exit)) {
                    $result = false;
                }
            }
            return $result;
        }

        if (!is_array($data) && !is_object($data)) {
            throw new Exception('Glb_Request::check_exists: first parameter must be an array');
        }

        if (!Glb_Hash::exists($data, $field_name)) {
            $result = false;
            $this->set_error( 400 , $failure_actions, $exit, ['field' => $field_name] );
        }
        return $result;
    }

    // @todo : check for potential hacks
    public function check_hacks($data, $field_name = null,
                                $failure_actions = ['status:set', 'notice:status', 'db_log:invalid_field', 'redirect:referer'],
                                $exit = true) {
        return true;
    }

    public function check_true($value, $field_name = null,
                               $failure_actions = ['status:set', 'notice:status', 'db_log:invalid_field', 'redirect:referer'],
                                    $exit = true) {
        $result = true;
        if (!$value) {
            $result = false;
            $this->set_error( 400 , $failure_actions, $exit, ['field' => $field_name] );
        }
        return $result;
    }

    /*public function check_not_empty_field($thingy, $field_name, $failure_actions = null, $exit = true) {
        $value = Glb_Hash::get($thingy, $field_name);
        if (empty($value)) {
            $this->set_error(400, $failure_actions, $exit, ['field' => $field_name]);
        }
        return true;
    }*/

    public function redirect($location, $status = 302) {
        if ($location == 'referer') {
            wp_redirect($this->get_referer(), $status);
        } else if ($location == 'home') {
            wp_redirect(home_url(), $status);
        } else if ($location == 'dashboard') {
            wp_redirect(get_dashboard_url(), $status);
        } else {
            wp_redirect($location, $status);
        }
        exit();
    }

}
