<?php

class Glb_Template
{

    private $template = null;
    private $plugin = null;
    private $type = null;

    /**
     * Glb_Template constructor. Used to render custom HTML elements.
     * @param string $template : file path. Can be optionally composed width subfolders and plugin key.
     * examples :
     *      'my-simple-template' will search in [caller plugin path]/templates/my-simple-template.php
     *      'gloubi-boulga:my-simple-template' will search in [gloubi-boulga plugin path]/templates/my-simple-template.php
     *      'gloubi-boulga.elements/my-simple-element-template' will search in [gloubi-boulga plugin path]/templates/elements/my-simple-template.php
     * @throws Exception
     */
    function __construct($template) {

        if (is_callable($template)) {

            $this->template = $template;
            $this->type = 'callable';

        } else {

            // remove starting slash if exists
            $template = Glb_Text::remove_leading($template, '/');
            //Glb_Log::info('searching for template ' . $template);

            // forbid empty
            if (empty($template)) {
                throw new Exception(__glb('Glb_Template : empty template can not be found'));
            }

            try {

                // explode and filter parts to get plugin
                $template_parts = array_filter(explode(':', $template), 'strlen');
                if (count($template_parts) == 1) {
                    $this->plugin = Glb_Plugin::get_caller_plugin();
                } else {
                    $this->plugin = Glb_Plugin::get_registered(array_shift($template_parts));
                }
                // read and search template
                $this->template = $template_parts[0];
                //Glb_Log::info('searching for template ' . $this->plugin->file_path('templates/' . $this->template . '.php'));

                if (!file_exists($this->plugin->file_path('templates/' . $this->template . '.php'))) {
                    throw new Exception(__glb('Glb_Template : Unable to find template « %s »', $this->template));
                }
                $this->type = 'file';

            } catch(Exception $ex) {
                throw new Exception(__glb('Glb_Template : Unable to find template « %s »', $this->template));
            }
        }

    }

    public function get_path() {
        return $this->plugin->file_path('templates/' . $this->template . '.php');
    }

    public function is_file() {
        return $this->type == 'file';
    }

    public function is_callable() {
        return $this->type == 'callable';
    }

    public function get_plugin() {
        return $this->plugin;
    }

    public static function get($template) {
        try {
            return new Glb_Template($template);
        } catch(Exception $ex) {
            return null;
        }
    }

    public function render($view_data = null, $return = false) {

        // enrich $view_data with plugin variable
        if ($view_data === null) {
            $view_data = [];
        }
        $view_data = array_merge(['plugin' => $this->plugin], $view_data);

        if ($this->type == "callable") {

            $result = call_user_func($this->template, $view_data);
            if ($return) {
                return $result;
            } else {
                echo $result;
            }

        } else if ($this->type == "file") {

            $result = Glb_Path::get_include_content($this->plugin->file_path('templates/' . $this->template . '.php'), $view_data);;
            if ($return) {
                return $result;
            } else {
                echo $result;
            }

        }
    }

}
