<?php


class Glb_Plugin extends Glb_Entity
{

    // all wordpress / gloubi plugins registered in a single variable
    protected static $_plugins = ['all' => [], 'registered' => []];

    public $config;
    public $database;
    public $settings;
    public $scripts;
    public $styles;
    public $buds;
    public $request;
    public $session;
    public $notices;

    // custom data, like auth_timeout
    protected $_data;

    /*
    * Glb_Plugin constructor : it will find by itself the plugin source infos, don't worry about that
    */
    function __construct() {

        // read plugin infos
        $plugin_infos = self::get_caller_plugin_infos();
        parent::__construct($plugin_infos);
        // read plugin config file.
        // First search for gloubi-boulga-config-my.server.com.php file
        $config = $plugin_infos->path . GLB_SEPARATOR . $plugin_infos->key . '-config-' . $_SERVER['HTTP_HOST'] . '.php';
        if (!file_exists($config)) {
            $config = $plugin_infos->path . GLB_SEPARATOR . $plugin_infos->key . '-config.php';
        }
        if (file_exists($config)) {
            $this->config = (include $config);
        }

        // register plugin
        self::_register($this);

        // build helpers
        $this->request = Glb_Request::instance();
        $this->database = Glb_Db::instance();
        $this->session = Glb_Session::instance();
        $this->notices = Glb_Notices::instance();

        $this->settings = new Glb_Plugin_Settings($this);
        $this->scripts = new Glb_Plugin_Scryptles('scripts', $this);
        $this->styles = new Glb_Plugin_Scryptles('styles', $this);
        $this->buds = new Glb_Plugin_Buds($this);
        $this->_data = new Glb_Collection();

        $this->add_action('wp_head',        [$this, '_run_front_head']);
        $this->add_action('admin_head',     [$this, '_run_admin_head']);
        $this->add_action('admin_menu',     [$this, '_run_admin_menu']);
        $this->add_action('init',           [$this, '_run_init']);
        $this->add_filter('body_class',     [$this, '_run_body_class'], 99);

        // include custom code
        // First search for gloubi-boulga-custom-my.server.com.php file
        $custom = [];
        $custom[] = $plugin_infos->path . GLB_SEPARATOR . $plugin_infos->key . '-custom-' . $_SERVER['HTTP_HOST'] . '.php';
        if (!file_exists($custom[0])) {
            $custom = [];
        }
        $custom[] = $plugin_infos->path . GLB_SEPARATOR . $plugin_infos->key . '-custom.php';
        foreach($custom as $custom_item) {
            if (file_exists($custom_item)) {
                include_once $custom_item;
            }
        }

    }

    /*
    * Run the plugin actions
    */
    public function run() {

        // run set_locale
        load_plugin_textdomain( $this->key, false, Glb_Path::concat([$this->key, 'langs']) );

        // load settings if needed
        $this->settings->load(empty($this->_settings) ? null : $this->_settings);

        // load buds if needed
        $this->buds->load(empty($this->_buds) ? null : $this->_buds);

        // run buds
        foreach($this->buds->items as $item) {
            $item->run();
        }

        // load scripts if needed
        $this->scripts->register(empty($this->_scripts) ? null : $this->_scripts);

        // load styles if needed
        $this->styles->register(empty($this->_styles) ? null : $this->_styles);

    }

    public function _run_init() {
        $this->run_routes();
    }

    public function render_template($template, $view_data = [], $return = false) {
        if (empty($view_data)) {
            $view_data = [];
        }
        $view_data = array_merge( ['plugin' => $this], $view_data);
        $template = Glb_Path::remove_leading_separator($template);

        // don't use Glb_Template::render to keep the $this variable available in template file
        $template = Glb_Template::get($template);
        extract($view_data);

        if ($template->is_file()) {

            ob_start();
            include $template->get_path();
            if ($return) {
                return ob_get_clean();
            } else {
                echo ob_get_clean();
            }

        } else if ($template->is_callable()) {

            if ($return) {
                return call_user_func($this->template, $view_data);
            } else {
                echo call_user_func($this->template, $view_data);
            }

        }
    }

    public function render_element($template, $view_data = [], $return = false) {
        $template = Glb_Path::remove_leading_separator($template);

        // don't use Glb_Template::render to keep the $this variable available in template file
        if (Glb_Text::starts_with($template, 'elements/')) {
            return $this->render_template($template, $view_data, $return);
        } else {
            return $this->render_template('elements/' . $template, $view_data, $return);
        }
    }

    public function render_page($template, $view_data = [], $return = false) {
        $template = Glb_Path::remove_leading_separator($template);

        // don't use Glb_Template::render to keep the $this variable available in template file
        if (Glb_Text::starts_with($template, 'pages/')) {
            return $this->render_template($template, $view_data, $return);
        } else {
            return $this->render_template('pages/' . $template, $view_data, $return);
        }
    }

    public function render_html($template, $element = [], $attrs = [], $options = [], $return = false) {

        // don't use Glb_Template::render to keep the $this variable access in template file
        $result = Glb_Html::get($template)->html($element, $attrs, $options);
        if (!$return) {
            echo $result;
        } else {
            return $result;
        }

    }

    public function render_json($data, $actions, $return = false) {
        $json = $this->request->build_json_response(200, null, $data, $actions);
        $this->request->set_response_header(['status' => 200]);

        if (!$return) {
            echo wp_json_encode($json);
            exit();
        } else {
            return wp_json_encode($json);
        }
    }

    /*public function enqueue_scripts() {
        // load scripts if needed
        $this->scripts->load(empty($this->_scripts) ? null : $this->_scripts);

        // load styles if needed
        $this->styles->load(empty($this->_styles) ? null : $this->_styles);
    }*/


    /*
     * @function register : add a GLB plugin to the registered array of plugins
     * @param string $plugin : the plugin object.
     * @return void
     */
    protected static function _register($plugin) {
        self::$_plugins['registered'][$plugin->key] = $plugin;
        self::_register_orm_hooks($plugin);
    }

    protected static function _register_orm_hooks($plugin) {
        $hooks_file = $plugin->file_path('models/hooks.php');
        if (file_exists($hooks_file)) {
            include_once $hooks_file;
        }
    }

    public static function get($plugin_key = null) {
        self::_check();
        $result = self::get_registered($plugin_key);
        if (empty($result)) {
            $result = self::get_all($plugin_key);
        }

        if (empty($result)) {
            if (Glb_Text::starts_with($plugin_key, 'glb-')) {
                $plugin_key = Glb_Text::remove_leading($plugin_key, 'glb-');
            } else {
                $plugin_key = Glb_Text::ensure_leading($plugin_key, 'glb-');
            }
            $result = self::get_registered($plugin_key);
            if (empty($result)) {
                $result = self::get_all($plugin_key);
            }
        }

        return $result;

    }

    public static function get_all($plugin_key = null) {

        self::_check();
        if ($plugin_key === null) {
            return self::$_plugins['all'];
        } else {
            if (array_key_exists($plugin_key, self::$_plugins['all'])) {
                return self::$_plugins['all'][$plugin_key];
            } else {
                foreach(self::$_plugins['all'] as $plugin) {
                    if ($plugin['alias'] == $plugin_key) {
                        return $plugin;
                    }
                }
                return null;
            }
        }
    }

    private static function _glb_metas($file) {
        $data = get_file_data($file, ['GlbAlias' => 'Glb Alias'], 'glb_plugin');
        return $data;
    }

    /**
     *
     * Check that the plugin list have already been loaded
     * If not, load it. For internal use only.
     *
     * @return bool True / false
     */
    private static function _check() {

        if (!empty(self::$_plugins['all'])) {
            return false;
        }

        // force WP get_plugins function to load if needed
        if ( ! function_exists( 'get_plugins' ) ) {
            require_once ABSPATH . 'wp-admin/includes/plugin.php';
        }

        // read all WP loaded plugins
        $all_plugins = get_plugins();

        foreach($all_plugins as $plugin_entry => $plugin_value) {

            // read plugin file to search for Internal name
            $plugin_key_split = explode('/', $plugin_entry);
            $glb_metas = self::_glb_metas(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $plugin_entry);
            if (empty($glb_metas['GlbAlias'])) {
                //$glb_metas['GlbAlias'] = str_replace('gloubi-boulga-', '', $plugin_key_split[0]);
                $glb_metas['GlbAlias'] = str_replace('glb-', '', $plugin_key_split[0]);
            }

            // build plugin Glb infos
            self::$_plugins['all'][$plugin_key_split[0]] = new Glb_Entity(Glb_Hash::underscorize_keys(
                array_merge($plugin_value,
                    [
                        'Key' => $plugin_key_split[0],
                        //'InternalKey' => str_replace('glb-', '', str_replace('gloubi-boulga-', '', $plugin_key_split[0])),
                        'Alias' => strtolower($glb_metas['GlbAlias']),
                        'AliasSource' => $glb_metas['GlbAlias'],
                        'Entry' => $plugin_key_split[1],
                        'Path' => Glb_Path::normalize(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $plugin_key_split[0]),
                        //'Config' => self::get_config_file($plugin_key_split),
                        'WpUri' => plugin_dir_url(WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $plugin_key_split[0] . '/index.php')
                    ] )
            )/*, ArrayObject::STD_PROP_LIST|ArrayObject::ARRAY_AS_PROPS*/);
        }

        return true;

    }

    protected static function _get_registered($plugin_key = null) {

    }

    /*
     * Get registered GLB plugin
     * @param string $plugin_key The plugin key. If null, then returns all registered Glb plugins.
     * @return array | Glb_Plugin | null : array of plugins infos, found Glb_Plugin or null if nothing was found
     */
    public static function get_registered($plugin_key = null) {

        self::_check();
        if ($plugin_key === null) {
            $result = self::$_plugins['registered'];
        } else {
            $result = Glb_Cache::get(__CLASS__ . ':registered:' . $plugin_key);
            if (empty($result)) {
                // @todo : clean that as plugin will be renamed
                // first, search in keys
                if (array_key_exists($plugin_key, self::$_plugins['registered'])) {
                    $result = Glb_Cache::set(__CLASS__ . ':registered:' . $plugin_key, self::$_plugins['registered'][$plugin_key]);
                } else {
                    // then, try to find using alias or glb-alias
                    foreach (self::$_plugins['registered'] as $plugin) {
                        if ($plugin['alias'] == $plugin_key || 'glb-' . $plugin['alias'] == $plugin_key) {
                            $result = Glb_Cache::set(__CLASS__ . ':registered:' . $plugin_key, $plugin);
                            break;
                        }
                    }
                }
            }
        }
        return $result;
    }

    /*
     * @function get_config_file : build config file path without checking existence
     */
    /*public static function get_config_file($plugin_keys) {
        if (!Glb_Text::ends_with($plugin_keys[1], '.php')) {
            return false;
        }
        return WP_PLUGIN_DIR . DIRECTORY_SEPARATOR . $plugin_keys[0] . DIRECTORY_SEPARATOR .
            Glb_Text::remove_trailing($plugin_keys[1], '.php') . '-config.php';
    }*/

    /*
     * @function get_caller_plugin_infos : return formatted array of plugin infos from where this
     * function has been called. eg :
     *    array (size=16)
     *      'name' => string 'Gloubi Boulga' (length=13)
     *      'plugin_uri' => string 'http://example.com/plugin-name-uri/' (length=35)
     *      'version' => string '1.0.0' (length=5)
     *      'description' => string 'Short description displayed in the admin are.'
     *      'author' => string 'Your Name or Your Company'
     *      'author_uri' => string 'http://example.com/'
     *      'text_domain' => string 'gloubi-boulga'
     *      'domain_path' => string '/languages'
     *      'network' => boolean false
     *      'title' => string 'Gloubi Boulga'
     *      'author_name' => string 'Your Name or Your Company'
     *      'key' => string 'gloubi-boulga'
     *      'entry' => string 'gloubi-boulga.php'
     *      'path' => string '/root/path-to-wordpress/wp-content/plugins/gloubi-boulga'
     *      'config' => string '/root/path-to-wordpress/wp-content/plugins/gloubi-boulga/gloubi-boulga-config.php'
     *      'wp_uri' => string 'http://localhost/glb/wp-content/plugins/gloubi-boulga/' (length=54)
     *
     */
    public static function get_caller_plugin_infos() {
        static $_cache = [];

        // load backtrace
        $bt = debug_backtrace();
        $bt = Glb_Array::get_stack();
        $plugin_key = 'glb-core';

        if (array_key_exists($cache_key = print_r($bt, true), $_cache)) {
            return self::get_all($_cache[$cache_key]);
        }

        // seach for last not glb-core gloubi boulga php file
        // (last and not first because a sub plugin can call Glb Core functions for ex.)
        //
        foreach($bt as $item) {
            $diff = Glb_Path::diff(WP_PLUGIN_DIR, $item);
            $temp = Glb_Path::explode($diff);
            $plugin_expected = $temp[0];
            if ($plugin_expected !== 'glb-core' && strpos($plugin_expected, 'glb-') === 0) {
                $plugin_key = $plugin_expected;
            }
        }

        // plugin key is the folder name
        if (!empty($plugin_key)) {
            $_cache[$cache_key] = $plugin_key;
            return self::get_all($plugin_key);
        }
        return null;
    }

    /**
     * @function get_caller_plugin : retrieves the plugin that called the function
     * @return Glb_Plugin
     */
    public static function get_caller_plugin()
    {
        $plugin_infos = self::get_caller_plugin_infos();
        return self::get($plugin_infos->key);
    }

    /*
    * @function get_config : returns the configuration value for the specified $key
    * @param $key : config item key
    * @param $default : default value to return if the key has not been found
    * @return string : the full local path where the $file can be accessed
    */
    public function get_config($key = null, $default = null) {
        if ($key === null) { return $this->config; }
        return (Glb_Hash::get($this->config, $key, $default));
        /*if (!empty($this->config) && array_key_exists($key, $this->config)) {
            return $this->config[$key];
        } else {
            return $default;
        }*/
    }

    /*
    * @function file_path : returns the local path for accessing the file
    * @param array | string $file : can be simple string or array of path parts from the glb plugin folder
    * @return string : the full local path where the $file can be accessed
    */
    public function file_path($file) {
        //$numargs = func_num_args();
        return Glb_Path::concat(array_merge([$this->path], func_get_args()));
    }

    /*
    * @function file_uri : returns the uri for downloading the file
    * @param array | string $file : can be simple string or array of path parts from the glb plugin folder
    * @return string : the full uri where the $file can be downloaded
    */
    public function file_uri($file) {
        if (is_string($file) && func_num_args() == 1 && strpos($file, '//') === 0) {
            return $file;
        }
        return Glb_Path::concat(array_merge([$this->wp_uri], func_get_args()), ['type' => 'uri']);
    }

    /*
     * @function dispatch_event : dispatch to all GLB plugins that implement event_$event_name function
     * @param string $source : plugin name
     * @param string $event_name : event name, eg 'cron'
     * @param mixed $event_args : arguments
     *
     */
    public function dispatch_event($source, $event_name, $event_args = null) {
        $plugins = Glb_Plugin::get_registered();
        $result = [];
        $real_event_name = 'event_' . $event_name;
        $source_key = is_string($source) ? $source : $source->key;

        foreach($plugins as $plugin) {
            if ($source_key != $plugin->key && method_exists($plugin, $real_event_name)) {
                $result[] = $plugin->$real_event_name($event_args);
            }
        }
    }

    /*
     * @function add_shortcode
     * @param string $shortcode : shortcode name
     * @param string $callback : callback, can be complex : [$object, 'function_name']
     */
    public function add_shortcode($shortcode, $callback) {
        add_shortcode($shortcode, $callback);
    }

    /*
     * @function add_route : adds a route (an URL) that redirects the request to a callback
     * @param string $name : route uri to detect
     * @param string $callback : callback to be called
     * @param string $params : params to be passed to the callback
     * @param string $places : Endpoint mask describing the places the endpoint should be added.
     * @param string $query_var : name of the corresponding query variable.
     *      Pass false to skip registering a query_var for this endpoint. Defaults to the value of $name.
     *      Default value: true
     */
    public function add_route($name, $callback, $params = null, $places = EP_ALL, $query_var = true) {
        $this->_data->set("routes.$name", [$name, $callback, $params, $places, $query_var]);
    }

    public function run_routes() {

        $routes = $this->_data->get("routes");
        if (!empty($routes)) {
            foreach($routes as $route_key => $route_value) {
                if (is_callable($route_value)) {
                    call_user_func($route_value, $route_key);
                } else {
                    add_rewrite_endpoint($route_value[0], $route_value[3], $route_value[4]);
                    $this->add_action( 'template_redirect', function() use ($route_value) {
                        global $wp_query;
                        //Glb_Log::notice('template redirect : ', $wp_query->query_vars);
                        if (isset($wp_query->query_vars[$route_value[0]])) {
                            call_user_func($route_value[1], $route_value[2]);
                        }
                    });
                }
            }
            flush_rewrite_rules();
        }

    }


    /*
     * @function add_action : adds an action using WP add_action function
     * @param string $tag : The name of the action to which the $callback is hooked.
     * @param callable $callback : callback to be called
     * @param int $priority : the priority of the function (lower is earlier). Default is 10.
     * @param int $accepted_args : the number of arguments the function accepts. Default is 1.
     * @return bool true
     */
    public function add_action($tag, $callback, $priority = 10, $accepted_args = 1) {
        return add_action($tag, $callback, $priority, $accepted_args);
    }

    /*
     * @function remove_action : removes a function from a specified action hook.
     * @param string $tag : the name of the action to which the $callback is hooked.
     * @param callable $callback : callback to be called
     * @param int $priority : the priority of the function (lower is earlier). Default is 10.
     * @return bool true
     */
    public function remove_action($tag, $callback, $priority = 10) {
        return remove_action($tag, $callback, $priority);
    }

    /*
     * @function add_filter : adds an action using WP add_action function
     * @param string $tag : The name of the action to which the $callback is hooked.
     * @param callable $callback : callback to be called
     * @param int $priority : the priority of the function (lower is earlier). Default is 10.
     * @param int $accepted_args : the number of arguments the function accepts. Default is 1.
     * @return bool true
     */
    public function add_filter($tag, $callback, $priority = 10, $accepted_args = 1) {
        return add_filter($tag, $callback, $priority, $accepted_args);
    }

    /*
     * @function remove_filter : adds an action using WP add_action function
     * @param string $tag : The name of the action to which the $callback is hooked.
     * @param callable $callback : callback to be called
     * @param int $priority : the priority of the function (lower is earlier). Default is 10.
     * @return bool true
     */
    public function remove_filter($tag, $callback, $priority = 10) {
        return remove_filter($tag, $callback, $priority);
    }

    /*
     * Run uninstallation process for the plugin
     * Must be called from the plugin's entry point
     * @return void
     */
    public static function uninstall() {
        $installer_class = get_class(self) . '_Installer';
        if (class_exists($installer_class, false)) {
            $installer = new $installer_class();
            $installer->uninstall();
        }
    }

    /*
     * Run activation process for the plugin
     * Must be called from the plugin's entry point
     * @return void
     */
    public static function activate() {
        $installer_class = get_called_class() . '_Installer';
        if (class_exists($installer_class, false)) {
            $installer = new $installer_class();
            $installer->activate();
        }
    }

    /*
     * Run deactivation process for the plugin
     * Must be called from the plugin's entry point
     * @return void
     */
    public static function deactivate() {
        $installer_class = get_called_class() . '_Installer';
        if (class_exists($installer_class, false)) {
            $installer = new $installer_class();
            $installer->deactivate();
        }
    }

    /*
     * @function add_script : registers a script that will be enqueued at the right moment
     * @return void
     */
    public function add_script($item) {
        $this->scripts->register( [$item] );
    }

    /*
     * @function add_style : registers a style that will be enqueued at the right moment
     * @return void
     */
    public function add_style($item) {
        $this->styles->register( [$item] );
    }

    public function _run_admin_menu($params) {
        $menus = $this->_data->get("admin_menus");
        if (!empty($menus)) {
            foreach($menus as $menu_key => $params) {
                add_menu_page($params[0], $params[1], $params[2], $params[3], $params[4], $params[5], $params[6]);
            }
        }
        $menus = $this->_data->get("admin_sub_menus");
        if (!empty($menus)) {
            foreach($menus as $menu_key => $params) {
                add_submenu_page($params[0], $params[1], $params[2], $params[3], $params[4], $params[5]);
            }
        }
    }

    /*
     * @return string : The resulting page's hook_suffix.
     */
    public function add_admin_menu($page_title, $menu_title, $capability, $menu_slug, $function = '', $icon_url = '', $position = null) {
        $this->_data->set("admin_menus.$menu_slug", [$page_title, $menu_title, $capability, $menu_slug, $function, $icon_url, $position]);
        //return add_menu_page($page_title, $menu_title, $capability, $menu_slug, $function, $icon_url, $position);
    }

    public function add_admin_sub_menu($parent_slug, $page_title, $menu_title, $capability, $menu_slug, $function = '', $position = null) {
        $this->_data->set("admin_sub_menus.$menu_slug", [$parent_slug, $page_title, $menu_title, $capability, $menu_slug, $function]);
    }

    protected function _run_head($type) {
        $headers = $this->_data->get("{$type}_headers");
        if (!empty($headers)) {
            foreach($headers as $header_key => $header_value) {
                if (is_callable($header_value)) {
                    echo call_user_func($header_value, $header_key);
                } else {
                    echo $header_value;
                }
            }
        }
    }

    public function _run_front_head() {
        $this->_run_head('front');
    }

    public function _run_admin_head() {
        $this->_run_head('admin');
    }

    public function add_front_header($key, $value) {
        $this->_data->set("front_headers.$key", $value);
    }

    public function remove_front_header($key) {
        $this->_data->remove_key("front_headers.$key");
    }

    public function add_admin_header($key, $value) {
        $this->_data->set("admin_headers.$key", $value);
    }

    public function remove_admin_header($key) {
        $this->_data->remove_key("admin_headers.$key");
    }

    public function add_both_header($key, $value) {
        $this->_data->set("front_headers.$key", $value);
        $this->_data->set("admin_headers.$key", $value);
    }

    public function remove_both_header($key) {
        $this->_data->remove_key("front_headers.$key");
        $this->_data->remove_key("admin_headers.$key");
    }

    public function add_body_class($class, $conditions = null) {
        if (empty($conditions) || $this->request->check_conditions($conditions)) {
            $body_classes = $this->_data->get('body_classes', []);
            $this->_data->set('body_classes', array_merge($body_classes, explode(' ', $class)));
        }
    }

    public function _run_body_class($classes) {
        if ( ($glb_classes = $this->_data->get('body_classes')) != null) {
            if (empty($classes)) { $classes = []; }
            return array_merge( $classes, $glb_classes );

        } else {
            return $classes;
        }
    }

}
