<?php
/**
 * Gloubi-Boulga Hash / Thingy Utility : a set of very dramatically usefull static functions
 */

class Glb_Hash
{

    /*
     * Find first existing key/property in $thingy
     *
     * @param array|object $thingy : haystack
     * @param array $keys Array of needles
     * @param mixed $default The default value to return if no keys were found
     * @param string $return The value | key | both : whether should return the key or the value, or both
     * @return string | array $return Value | key | both : return the key or the value or both
     *
     * @usage :
     *
     *      Glb_Hash::first_key(['mum' => '0', 'dad' => 1, 'postman' => 2], ['evelyne', 'john', 'postman'], null, 'value')
     *          returns 2
     *
     *      Glb_Hash::first_key(['mum' => '0', 'dad' => 1, 'postman' => 2], ['evelyne', 'john', 'postman'], null, 'key')
     *          returns 'postman'
     *
     *      Glb_Hash::first_key(['mum' => '0', 'dad' => 1, 'postman' => 2], ['evelyne', 'john', 'postman'], null, 'both')
     *          returns ['key' => 'postman', 'value' => 2]
     */
    public static function first($thingy, $keys, $return = 'value', $default = null) {

        if (!self::_check_thingy($thingy)) { return $default; }
        if (empty($thingy) || empty($keys)) { return $default; }
        foreach ($keys as $key) {
            if (self::exists($thingy, $key)) {
                if ($return == 'value') {
                    return self::_get($thingy, $key);
                } else if ($return == 'key') {
                    return $key;
                } else if ($return == 'both') {
                    return ['key'  => $key, 'value' => self::_get($thingy, $key)];
                }
            }
        }
        return $default;
    }

    /**
     * Test key/property existency in $thingy, at first !evel
     * @param object|array $thingy The haystack
     * @param string $key
     * @return bool
     */
    public static function _exists($thingy, $key) {
        $test = self::attributes($thingy);
        return array_key_exists($key, $test);
    }

    /**
     * Returns keys/properties found in $thingy, at first !evel
     * @param object|array $thingy The haystack
     * @return array
     */
    public static function keys($thingy) {
        return array_keys(self::attributes($thingy));
    }

    /**
     * Test key/property existency in $thingy
     *
     * @param object|array $thingy The haystack
     * @param string $key Can be a key path such as 'key.subkey'...
     * @return bool The found result (purely logical)
     *
     */
    public static function exists($thingy, $key) {

        if (empty($thingy)) { return false; }

        if (!is_array($key)) {
            $key_parts = explode('.', $key);
        } else {
            $key_parts = $key;
        }
        $current = $thingy;
        for($i = 0; $i < count($key_parts); $i++) {
            if (self::_exists($current, $key_parts[$i])) {
                if ($i == count($key_parts)-1) {
                    return true;
                } else {
                    $current = self::_get($current, $key_parts[$i]);
                }
            } else {
                return false;
            }
        }
        return false;
    }

    /**
     * Test if $thingy's key/property is empty or not
     *
     * @param object|array $thingy The haystack
     * @param string $key Can be a key path such as 'key.subkey'...
     * @return bool True if empty (wich is not so stupid)
     */
    public static function is_empty($thingy, $key) {
        return empty(self::get($thingy, $key));
    }

    /**
     * Get the value with no existency check before ! Internal use only.
     * @param array|object $thingy The haystack
     * @param string $key Can NOT be a key path such as 'key.subkey'...
     * @return mixed : found value
     */
    private static function _get($thingy, $key) {

        if (is_array($thingy) || is_a($thingy, 'ArrayObject')) {
            return $thingy[$key];
        } else if (is_object($thingy)) {
            return $thingy->$key;
        }
    }

    private static function _check_thingy($thingy) {
        if (!is_array($thingy) && !is_object($thingy)) {
            return false;
        } else {
            return true;
        }
    }

    /*
     * Find key/property in $thingy and returns value
     * @param array|object $thingy The haystack
     * @param string $key Can be a key path such as 'key.subkey'...
     * @param mixed $default : the default value to return if key was not found
     * @return mixed : the value found, or $default if nothing was found
     *
     * @usage :
     *
     *      Glb_Hash::get(['mum' => '0', 'dad' => 1, 'postman' => ['child1', 'child2']], 'postman.0')
     *          returns 'child1'
     */

    public static function get($thingy, $key, $default = null) {

        if (!self::_check_thingy($thingy)) { return $default; }
        if (empty($thingy) || empty($key)) { return $default; }

        if (!is_array($key)) {
            $key_parts = explode('.', $key);
        } else {
            $key_parts = $key;
        }

        $current = $thingy;
        foreach($key_parts as $key) {
            if (self::exists($current, $key)) {
                $current = self::_get($current, $key);
            } else {
                return $default;
            }
        }

        return $current;
    }

    /*
     * A way to create quickly deep thingy in one line of code
     * @param array $values An associative array to set keys/values
     * @param (optional) $type The type of the return (array, ArrayObject, stdclass...)
     *
     * @usage
     *
     * Glb_Hash::build(['element.name' => 'ze_name', 'element.key' => 'ze_key', 'attrs.class' => 'ze_class'], 'array');
     *      returns ['element' => ['name' => 'ze_name', 'key' => 'ze_key'], 'attrs' => ['class' => 'ze_class']]]
     *
     * Glb_Hash::build(['element.name' => 'ze_name', 'element.key' => 'ze_key', 'attrs.class' => 'ze_class'], 'ArrayObject');
     *      returns new ArrayObject(['element' => ['name' => 'ze_name', 'key' => 'ze_key'], 'attrs' => ['class' => 'ze_class']] ])
     *
     */
    public static function build($values, $type = 'array') {

        $thingy = null;
        $ltype = strtolower($type);
        if ($type == 'array') {
            $thingy = [];
        } else if ($type == 'glb_entity') {
            $thingy =  new Glb_Entity([]);
        } else {
            $class = $type;
            $thingy =  new $class();
        }
        foreach($values as $key => $value) {
            self::set($thingy, $key, $value);
        }
        return $thingy;
    }


    /*
     * Set value at the specified $key (overwrites the eventually already existing item)
     *
     * @param array|object $thingy Haystack
     * @param array $key Needle key, can be a path such as 'parent_key.child_key'
     * @param mixed $value The default value to return if key was not found
     * @return mixed The updated par of thingy (full thingy modified by reference)
     *
     * @usage :
     *
     *      $array = ['mum' => '0', 'dad' => 1];
     *      Glb_Hash::set($array, 'postman.child1', 'unknown')
     *          returns ['mum' => '0', 'dad' => 1, 'postman' => ['child1' => 'unknown']]
     *
     *      $array = ['mum' => '0', 'dad' => 1];
     *      Glb_Hash::set($array, 'dad', '2')
     *          returns ['mum' => '0', 'dad' => 2]
     *
     *      $array = ['mum' => '0', 'dad' => 1];
     *      Glb_Hash::set($array, 'dad', ['grandpa' => 3]])
     *          returns ['mum' => '0', 'dad' => ['grandpa' => 3]]
     */
    public static function set(&$thingy, $key, $value) {

        $key_parts = explode('.', $key, 2);
        if (count($key_parts) == 1) {
            return self::_set($thingy, $key_parts[0], $value, false);
        } else if (!self::exists($thingy, $key_parts[0])) {
            self::_set($thingy, $key_parts[0], []);

            if (is_array($thingy) || is_a($thingy, 'ArrayObject')) {
                return self::set($thingy[$key_parts[0]], $key_parts[1], $value);
            } else if (is_object($thingy)) {
                return self::set($thingy->{$key_parts[0]}, $key_parts[1], $value);
            } else {
                return null;
            }

            /*
            if (is_array($thingy)) {
                self::_set($thingy, $key_parts[0], []);
                return self::set($thingy[$key_parts[0]], $key_parts[1], $value);
            } else if (is_a($thingy, 'ArrayObject')) {
                $class = get_class($thingy);
                self::_set($thingy, $key_parts[0], new $class([]));
                return self::set($thingy[$key_parts[0]], $key_parts[1], $value);
            } else if (is_object($thingy)) {
                $class = get_class($thingy);
                self::_set($thingy, $key_parts[0], new $class());
                return self::set($thingy->{$key_parts[0]}, $key_parts[1], $value);
            }*/
        } else {
            if (is_array($thingy) || is_a($thingy, 'ArrayObject')) {
                return self::set($thingy[$key_parts[0]], $key_parts[1], $value);
            } else if (is_object($thingy)) {
                return self::set($thingy->{$key_parts[0]}, $key_parts[1], $value);
            } else {
                return null;
            }

        }

    }

     /*
     * Set value at the specified $key, MERGING with the eventually already existing item
     * WARNING !! If you use that, please make sure that the final item type is not an
     * array in its original form, otherwise it could give something very very weird.
     * If you need an Array like item, please use ArrayObject or Glb_Entity
     *
     * @param array|object $thingy : haystack
     * @param string $key : needle key, can be a path such as 'parent_key.child_key'
     * @param mixed $value : the default value to return if key was not found
     * @return mixed : the modified part of $thingy ($thingy is modified by reference)
     *
     * @usage :
     *
     *      $array = ['mum' => '0', 'dad' => 1];
     *      Glb_Hash::append($array, 'dad', '2')
     *          returns [1, 2]
     *          while $thingy becomes ['mum' => '0', 'dad' => [1, 2]]
     *
     *      $array = ['mum' => '0', 'dad' => 1];
     *      Glb_Hash::append($array, 'dad', ['grandpa' => 3]])
     *          returns [1, ['grandpa' => 3]]
     *          while $thingy becomes ['mum' => '0', 'dad' => [1, ['grandpa' => 3]]]
     *
     *      $array = ['mum' => '0'];
     *      Glb_Hash::append($array, 'dad.has.teeth', 32)
     *          returns 32
     *          while $thingy becomes ['mum' => '0', 'dad' => ['has' => ['teeth' => 32]]]
     *
     *      Glb_Hash::append($array, 'dad.has.teeth', 'allwhite')
     *          returns [32, 'allwhite']
     *          while $thingy becomes ['mum' => '0', 'dad' => ['has' => ['teeth' => [32, 'allwhite']]]]
     *
     *
     */
    public static function append(&$thingy, $key, $value) {

        $key_parts = explode('.', $key, 2);

        if (count($key_parts) == 1) {

            // if only one key found (not a path), just set with merge option
            return self::_set($thingy, $key_parts[0], $value, true);

        } else if (self::exists($thingy, $key_parts[0])) {

            if (is_array($thingy) || is_a($thingy, 'ArrayObject')) {
                return self::_set($thingy[$key_parts[0]], $key_parts[1], $value, true);
            } else if (is_object($thingy)){
                return self::_set($thingy->{$key_parts[0]}, $key_parts[1], $value, true);
            }

        } else {

            // otherwise, create the key till the end
            if (is_array($thingy) || is_a($thingy, 'ArrayObject')) {
                $thingy[$key_parts[0]] = [];
                //return self::_set($thingy[$key_parts[0]], $key_parts[1], $value, true);
                return self::append($thingy[$key_parts[0]], $key_parts[1], $value);
            } elseif (is_object($thingy)) {
                $thingy->{$key_parts[0]} = [];
                return self::append($thingy->{$key_parts[0]}, $key_parts[1], $value);
            }

            /*if (is_array($thingy)) {
                return self::_set($thingy, $key_parts[0], [], true);
                return self::append($thingy[$key_parts[0]], $key_parts[1], $value);
            } elseif (is_a($thingy, 'ArrayObject')) {
                return self::_set($thingy, $key_parts[0], new ArrayObject([]), true);
                return self::append($thingy[$key_parts[0]], $key_parts[1], $value);
            } else {
                $class = get_class($thingy);
                return self::_set($thingy, $key_parts[0], new $class(), true);
                return self::append($thingy->{$key_parts[0]}, $key_parts[1], $value);
            }*/
        }
    }


    /**
     * Set value to thingy, with no key existency before ! Internal use only.
     *
     * @param array|object $thingy The haystack
     * @param string $key Can be a key path such as 'key.subkey'...
     * @param mixed $value The value to set
     */
    private static function _set(&$thingy, $key, $value, $merge = false)
    {

        $value2 = $value;
        if ($merge && self::exists($thingy, $key)) {
            $value2 = self::get($thingy, $key);
            if (!is_array($value2)) {
                $value2 = [$value2];
            }
            $value2[] = $value;
        }

        if (is_array($thingy) || is_a($thingy, 'ArrayObject')) {
            return $thingy[$key] = $value2;
        } else if (is_object($thingy)) {
            return $thingy->$key = $value2;
        }
    }

    /**
     *
     * Remove an item and get it in return. Internal use only.
     *
     * @param array|object $thingy The haystack
     * @param array $keyparts The full key_parts
     * @param integer $key_index The key index to remove in the previous key_parts array
     * @param mixed $default The default value to return if item not found
     * @return mixed|null The removed item or NULL
     */
    private static function _remove(&$thingy, $keyparts, $key_index, $default = null)
    {
        $result = $default;

        $current = &$thingy;
        for ($i = 0; $i < $key_index; $i++) {
            if (is_array($current) || is_a($current, 'ArrayObject')) {
                $current = &$current[$keyparts[$i]];
            } else {
                $current = &$current->$keyparts[$i];
            }
        }
        if (is_array($current) || is_a($current, 'ArrayObject')) {
            $result = $current[$keyparts[$i]];
            unset($current[$keyparts[$i]]);
        } else {
            $result = clone $current->{$keyparts[$i]};
            unset($current->{$keyparts[$i]});
        }
        return;

        if ($key_index > 12) {
            return null;
            //throw new Exception("Glb_Hash: can not remove at this depth ($key_index), sorry...");
        }

        if (is_array($thingy) || is_a($thingy, 'ArrayObject')) {
            switch ($key_index) {
                case 0:
                    $result = ($thingy[$keyparts[0]]);
                    unset($thingy[$keyparts[0]]);
                    break;
                case 1:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]]);
                    break;
                case 2:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]]);
                    break;
                case 3:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]]);
                    break;
                case 4:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]]);
                    break;
                case 5:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]]);
                    break;
                case 6:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]]);
                    break;
                case 7:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]]);
                    break;
                case 8:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]]);
                    break;
                case 9:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]][$keyparts[9]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]][$keyparts[9]]);
                    break;
                case 10:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]][$keyparts[9]][$keyparts[10]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]][$keyparts[9]][$keyparts[10]]);
                    break;
                case 11:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]][$keyparts[9]][$keyparts[10]][$keyparts[11]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]][$keyparts[9]][$keyparts[10]][$keyparts[11]]);
                    break;
                case 12:
                    $result = ($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]][$keyparts[9]][$keyparts[10]][$keyparts[11]][$keyparts[12]]);
                    unset($thingy[$keyparts[0]][$keyparts[1]][$keyparts[2]][$keyparts[3]][$keyparts[4]][$keyparts[5]][$keyparts[6]][$keyparts[7]][$keyparts[8]][$keyparts[9]][$keyparts[10]][$keyparts[11]][$keyparts[12]]);
                    break;
            }
        } else if (is_object($thingy)) {

            switch ($key_index) {
                case 0:
                    $result = ($thingy->{$keyparts[0]});
                    unset($thingy->{$keyparts[0]});
                    break;
                case 1:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]});
                    break;
                case 2:
                    $result = self::get("0.1.2");
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]});
                    break;
                case 3:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]});
                    break;
                case 4:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]});
                    break;
                case 5:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]});
                    break;
                case 6:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]});
                    break;
                case 7:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]});
                    break;
                case 8:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]});
                    break;
                case 9:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]}->{$keyparts[9]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]}->{$keyparts[9]});
                    break;
                case 10:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]}->{$keyparts[9]}->{$keyparts[10]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]}->{$keyparts[9]}->{$keyparts[10]});
                    break;
                case 11:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]}->{$keyparts[9]}->{$keyparts[10]}->{$keyparts[11]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]}->{$keyparts[9]}->{$keyparts[10]}->{$keyparts[11]});
                    break;
                case 12:
                    $result = ($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]}->{$keyparts[9]}->{$keyparts[10]}->{$keyparts[11]}->{$keyparts[12]});
                    unset($thingy->{$keyparts[0]}->{$keyparts[1]}->{$keyparts[2]}->{$keyparts[3]}->{$keyparts[4]}->{$keyparts[5]}->{$keyparts[6]}->{$keyparts[7]}->{$keyparts[8]}->{$keyparts[9]}->{$keyparts[10]}->{$keyparts[11]}->{$keyparts[12]});
                    break;
            }
        }

        return $result;

    }

    /**
     * Remove an item of $thingy and return the removed value
     *
     * @param array|object $thingy : haystack
     * @param array|string $key Needle key, can be a path such as 'parent_key.child_key'
     * @param mixed $default The default value to return if key's not found
     * @return mixed The removed value (the new thingy is modified by reference)
     */
    public static function remove(&$thingy, $key, $default = null) {

        if (empty($thingy) || empty($key)) {
            return false;
        }

        if (!is_array($key)) {
            $key_parts = explode('.', $key);
        } else {
            $key_parts = $key;
        }

        //----------
        $result = $default;

        $current = &$thingy;
        for ($i = 0; $i < count($key_parts) - 1; $i++) {
            if (!self::exists($current, $key_parts[$i])) { return $default; }
            if (is_array($current) || is_a($current, 'ArrayObject')) {
                $current = &$current[$key_parts[$i]];
            } else if (is_object($current)) {
                $current = &$current->{$key_parts[$i]};
            } else {
                return $default;
            }
        }

        $x = Glb_Text::unique_id('~~##~~!!~~##~~');
        $removed = self::get($current, $key_parts[$i], $x);
        if ($removed === $x) { return $default; }

        if (is_array($current) || is_a($current, 'ArrayObject')) {
            unset($current[$key_parts[$i]]);
        } else if (is_object($current)) {
            unset($current->{$key_parts[$i]});
        }

        return $removed;
    }

    /*
     * Test if an array or object contains an empty value
     * @param array|object $thingy : the thingy to evaluate
     * @param array $keys : the keys/properties array, can be key path
     * @return true if one of the keys doesn't exist or if one of the values found is empty
     */
    public static function one_empty($thingy, $keys = null) {
        if (empty($keys)) {
            $keys = array_keys(self::attributes($thingy));
        }

        if (empty($thingy)) {
            return true;
        }
        foreach($keys as $key) {
            if (empty(self::get($thingy, $key))) {
                return true;
            }
        }
        return false;
    }

    /*
     * Test if an array or object contains unset keys/properties
     * @param array|object $thingy : the thingy to evaluate
     * @param array $keys : the keys/properties array, can be key path
     * @return true if one of the keys doesn't exist (not empty testing)
     */
    public static function one_not_set($thingy, $keys) {
        if (empty($thingy) || empty($keys)) {
            return true;
        }
        foreach($keys as $key) {
            if (!self::exists($thingy, $key)) {
                return true;
            }
        }
        return false;
    }

    /*
     * Underscorize all keys at first level for an array or object
     *
     * @param array|object $thingy The thing to modify
     * @return array|object The modified $thingy
     *
     * @usage :
     *
     *      Glb_Hash::underscorize_keys( ['mumDad' => 'myParents', 'post-Man name$x' => 'theGuy'] )
     *          returns ['mum_dad' => 'myParents', 'post_man_name_x' => 'theGuy']
     */
    public static function underscorize_keys($thingy) {
        if (is_array($thingy)) {
            $result = [];
        } else if (is_object($thingy)) {
            $class = get_class($thingy);
            $result = new $class([]);
        } else {
            return null;
        }

        $attributes = self::attributes($thingy);
        foreach($attributes as $attribute_key => $attribute_value) {
            $attribute_key = str_replace('URI', 'Uri', $attribute_key);  // to avoid _u_r_i
            $attribute_key = str_replace('PHP', 'Php', $attribute_key);  // to avoid _u_r_i
            $attribute_key = str_replace('WP', 'Wp', $attribute_key);  // to avoid _u_r_i
            self::set($result, Glb_Text::underscore($attribute_key), $attribute_value);
        }

        return $result;
    }

    public static function lowerize_keys($thingy) {
        if (is_array($thingy)) {
            $result = [];
        } else if (is_object($thingy)) {
            $class = get_class($thingy);
            $result = new $class([]);
        } else {
            return null;
        }

        $attributes = self::attributes($thingy);
        foreach($attributes as $attribute_key => $attribute_value) {
            $attribute_key = mb_strtolower($attribute_key);  // to avoid _u_r_i
            self::set($result, Glb_Text::underscore($attribute_key), $attribute_value);
        }

        return $result;
    }

    /**
     * Check key existency. If keys doesn't exist, then fill with $values.
     *
     * @param array|object $thingy
     * @param array $keys Array of $keys to search for
     * @param mixed|array $values Value or array of values to set if keys not found
     *      If array, then all not existing keys are expected to be set with the same-indexed item of $values
     * @return mixed Modified $thingy (also modified by reference)
     *
     * @usage :
     *
     *      Glb_Hash::ensure_values( ['key1' => 'value1', 'key2' => 'value2'], ['key1', 'key3'], 'missing' )
     *          returns ['key1' => 'value1', 'key2' => 'value2', 'key3' => 'missing']
     *
     *      Glb_Hash::ensure_values( ['key1' => 'value1', 'key2' => 'value2'], ['key1', 'key2', 'key3'], ['missing1', 'missing2', 'missing3'] )
     *          returns ['key1' => 'value1', 'key2' => 'value2', 'key3' => 'missing3']
     */
    public static function ensure_values(&$thingy, $keys, $values = null) {
        if (!is_array($keys)) {
            $keys = [$keys];
            $values = [$values];
        }
        //Glb_Array::ensure($keys);
        //Glb_Array::ensure($values);
        foreach($keys as $key_index => $key) {
            if (!self::exists($thingy, $key)) {
                if (is_array($values)) {
                    self::set($thingy, $key, $values[min($key_index, count($values)-1)]);
                } else {
                    self::set($thingy, $key, $values);
                }
            }
        }
        return $thingy;
    }

    protected static function _find_pattern($thingy, $search_by, $pattern, $count = 1, $keys = null, $return = 'value', $default = null) {
        $result = [];
        if (empty($thingy)) {
            return $thingy;
        } else {
            if ($keys === null) {
                $keys = self::keys($thingy);
            }
            foreach($keys as $key) {
                if ($search_by == 'value') {
                    $compare = self::get($thingy, $key);
                } else {
                    $compare = $key;
                }
                if (is_scalar ($compare)) {
                    if (preg_match($pattern, $compare)) {
                        if ($return == 'value') {
                            $result[] = self::get($thingy, $key);
                        } else if ($return == 'key') {
                            $result[] = $key;
                        } else {
                            $result[] = ['key' => $key, 'value' => self::get($thingy, $key)];
                        }
                    }
                }
                if ($count == count($result)) {
                    break;
                }
            }
            if (empty($result)) {
                return $default;
            } else {
                return $result;
            }
        }
    }

    /**
     * Find an item fitting a regex pattern, returns the $count first fitting items found (value / key or both)
     * @param array|object $thingy
     * @param string $pattern Regex pattern to search for
     * @param int $count Max result count to return
     * @param array $keys Array of $keys to search for
     * @param mixed $default The default value to return if nothing found
     * @return mixed The found value(s) as an array
     * @usage
     *
     * Glb_Hash::find_value_pattern( ['key1' => 'Devil', 'key2' => 'Satan', 'key3' => '666'], '/^\d+$/', ['key0', 'key1', 'key2', 'key3'] )
     *      returns '666'
     *
     */

    public static function find_value_pattern($thingy, $pattern, $count = 0, $keys = null, $return = 'value', $default = null)
    {
        return self::_find_pattern($thingy, 'value', $pattern, $count, $keys, $return, $default);
    }

    /**
     * Find items fitting a regex pattern, returns the $count first fitting items found  (value / key or both)
     * @param array|object $thingy
     * @param string $pattern Regex pattern to search for
     * @param int $count Max result count to return (0 returns everything)
     * @param array $keys Array of $keys to search for
     * @param mixed $default The default value to return if nothing found
     * @return array The found value(s) as an array
     * @usage
     *
     * Glb_Hash::find_pattern( ['key1' => 'Devil', 'key2' => 'Satan', 'key3' => '666'], '/^\d+$/', ['key0', 'key1', 'key2', 'key3'] )
     *      returns '666'
     *
     */

    public static function find_key_pattern($thingy, $pattern, $count = 0, $keys = null, $return = 'value', $default = null)
    {
        return self::_find_pattern($thingy, 'key', $pattern, $count, $keys, $return, $default);
    }

    /**
     * Get all the keys/properties with associated values defined in $thingy
     * Use full if $thingy is an objet -> returns an array
     *
     * @param array|object $thingy
     * @return array Array of keys/properties found
     *
     * @usage
     *
     * Glb_Hash::attributes( (object)['key1' => 'Devil', 'key2' => 'Satan', 'key3' => '666']) )
     *      returns ['key1' => 'Devil', 'key2' => 'Satan', 'key3' => '666']
     *
     */
    public static function attributes($thingy) {

        if (empty($thingy)) { return []; }

        if ($thingy instanceof Glb_Entity) {
            return $thingy->to_array();
        } else if (is_array($thingy)) {
            return $thingy;
        } else if (is_a($thingy, 'ArrayObject')) {
            return $thingy->getArrayCopy();
        } else if (is_object($thingy)) {
            return get_object_vars($thingy);
        } else {
            return [];
        }

    }


    /**
     * Duplicate an array|object item
     * @param array|object $thingy
     * @return mixed The cloned object/array
     */
    public static function copy($thingy) {
        if (is_array($thingy)) {
            return $thingy;
        } else {
            return clone $thingy;
        }
    }

    /**
     * Merge two object|array items
     * @param array|object $thingy1
     * @param array|object $thingy2
     * @return mixed
     */
    public static function merge($thingy1, $thingy2) {

        $result = self::copy($thingy1);
        $props2 = self::attributes($thingy2);

        foreach($props2 as $prop2 => $val2) {
            self::set($result, $prop2, $val2);
        }

        return $result;
    }



    /*
     * Find key/property/functions in $thingy and returns value
     * This function searches in keys, properties and FUNCTIONS !
     * The search with function names is the only différence with the ::get function
     *
     * @param array|object $thingy The haystack
     * @param string $key Can be a key path such as 'key.subkey'...
     * @param mixed $default The default value to return if key was not found
     * @return mixed The value found, or $default if nothing was found
     *
     * @usage
     *
     * class Foo {
     *      public function my_function() {
     *          return new Sub_Foo();
     *      }
     *  }
     *
     * class Sub_Foo {
     *      public function my_sub_function() {
     *          return ['youhou' => 'I am here'];
     *      }
     *  }
     *
     *  Glb_Hash->get_stuff(new Foo(), 'my_function.my_sub_function')
     *      returns ['youhou' => 'I am here']
     *
     */

    public static function get_stuff($thingy, $key, $default = null) {

        if (empty($thingy) || empty($key)) {
            return $default;
        }

        if (!is_array($key)) {
            $key_parts = explode('.', $key);
        } else {
            $key_parts = $key;
        }

        $current = $thingy;
        foreach($key_parts as $key) {
            if (self::exists($current, $key)) {
                $current = self::_get($current, $key);
            } else if (method_exists($current, $key)) {
                $current = call_user_func([$current, $key]);
            } else {
                return $default;
            }
        }

        return $current;
    }

    /**
     * Flattens a nested array to a flat array.
     * @param mixed $thingy Array|Glb_Entity|ArrayObject|Array to flatten
     * @param string $separator The separator character to use in flattenization process
     * For example, flattens an array that was expanded with Glb_Hash::unflatten() into a flat array.
     * So, [['Super' => ['Man' => 'IsTheBest']]] becomes ['0.Super.Man' => 'IsTheBest'].
     */
    public static function flatten($thingy, $separator = '.') {

        $result = [];
        $attributes = static::attributes($thingy);
        $stack = [];
        $path = null;

        reset($attributes);
        while (!empty($attributes)) {
            $key = key($attributes);
            $element = $attributes[$key];
            unset($attributes[$key]);

            // if $element is a hierarchical thingy, then process it now,
            // and register $attributes to continue with it later
            if (static::_check_thingy($element) && !empty($element)) {
                if (!empty($attributes)) {
                    $stack[] = [$attributes, $path];
                }
                $attributes = $element;
                reset($attributes);
                $path .= $key . $separator;
            } else {
                $result[$path . $key] = $element;
            }

            if (empty($attributes) && !empty($stack)) {
                list($attributes, $path) = array_pop($stack);
                reset($attributes);
            }
        }

        return $result;
    }

    /**
     * Expands a flat array to a nested array.
     * @param array|object $thingy to unflatten
     * @param string $separator The separator character to use in flattenization process
     * For example, unflattens an array that was collapsed with Glb_Hash::flatten() into a multi-dimensional array.
     * So, ['0.Super.Man' => 'IsTheBest'] becomes [['Super' => ['Man' => 'IsTheBest']]].
     */
    public static function unflatten($thingy, $separator = '.') {
        $result = [];
        $attributes = static::attributes($thingy);
        foreach ($attributes as $key => $value) {
            Glb_Hash::set($result, $key, $value);
        }
        return $result;
    }

}
