<?php
/**
 * FlorianWolters\Component\Core\StringUtils
 *
 * PHP Version 5.3
 *
 * @author    Florian Wolters <wolters.fl@gmail.com>
 * @copyright 2010-2014 Florian Wolters (http://blog.florianwolters.de)
 * @license   https://gnu.org/licenses/lgpl.txt LGPL-3.0+
 * @link      https://github.com/FlorianWolters/PHP-Component-Core-StringUtils
 */
//namespace FlorianWolters\Component\Core;
/**
 * The class {@see StringUtils} offers operations `string`s.
 *
 * This class is inspired by the Java class {@link
 * https://commons.apache.org/proper/commons-lang/javadocs/api-3.1/org/apache/commons/lang3/StringUtils.html
 * StringUtils} from the {@link https://commons.apache.org/lang Apache Commons
 * Lang Application Programming Interface (API)}.
 *
 * @since Class available since Release 0.1.0
 */

if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN')
    define("GLB_SEPARATOR", "\\");
//define("GLB_SEPARATOR", "/");
else
    define("GLB_SEPARATOR", "/");


final class Glb_Path
{

    /**
     * Means : if you want to replace back or forward slashes with the occurate separator
     * @param string $path The path to normalize
     * @param bool $file_system If true, then the directory separator will be used. If false, then the forward slash will be used.
     * @return mixed
     */
    public static function normalize($path, $file_system = true) {
        $path = str_replace('/', $file_system ? GLB_SEPARATOR : '/', $path);
        $path = str_replace('\\', $file_system ? GLB_SEPARATOR : '/', $path);
        if ($file_system) {
            $path = str_replace('//', $file_system ? GLB_SEPARATOR : '/', $path);
            $path = str_replace('\\\\', $file_system ? GLB_SEPARATOR : '/', $path);
        }
        return $path;
    }

    /**
     * Ensure folder existency / permissions : creates it if doesn't exist
     * @param $folder
     * @param int $mode
     */
    public static function ensure_folder($folder, $mode = 0777) {
        if (!is_dir($folder)) {
            mkdir($folder, $mode, true);
        } else {
            chmod($folder, $mode);
        }
    }

    /**
     * Ensure that a 100% deny .htaccess file exists in a folder. Creates it otherwiser.
     * @param string $folder
     * @param string $duration If the file is older than $duration, then recreate it (+1 day for example)
     */
    public static function ensure_htdeny($folder, $duration = '+999 years') {

        if (self::is_older_than(self::concat([$folder, '.htaccess']), $duration)) {
            $htaccess_content =
                '# Please do not edit this file, or you are dead...' . "\r\n" .
                '<FilesMatch ".*">' . "\r\n" .
                'Order Allow,Deny' . "\r\n" .
                'Deny from All' . "\r\n" .
                '</FilesMatch>';
            file_put_contents(self::concat([$folder, '.htaccess']), $htaccess_content);
        }

    }

    /**
     * Calculates the difference between a file mtime and (now - $duration). Il a file doesn't exist, return true.
     * @param string $path
     * @param string $duration If the file is older than $duration, then recreate it (+1 day for example)
     */
    public static function is_older_than($path, $duration) {

        if (!file_exists($path) || (($created = filemtime($path)) === false) || empty($duration)) {
            return true;
        }

        $compare = strtotime($duration) - time();
        $datediff = time() - $created;
        return ($datediff > $compare);
    }

    /**
     * Explode a path into an array of folders, using forward and back slashed as separators
     * @param string $path
     * @return array[]|string[]
     */
    public static function explode($path) {
        return preg_split( '/[\\\\\/]/', $path);
    }

    /* compare two paths from the beginning and returns the common base */
    public static function intersect($path1, $path2, $file_system = true) {
        $path1 = self::explode($path1);
        $path2 = self::explode($path2);
        $result = [];
        foreach($path1 as $index => $value) {
            if ($value === $path2[$index]) {
                $result[] = $value;
            } else {
                break;
            }
        }
        return implode($file_system ? GLB_SEPARATOR : '/', $result);
    }

    /* compare two paths from the beginning and returns the difference */
    public static function diff($path1, $path2, $file_system = true) {
        if (strlen($path1) > strlen($path2)) {
            $longuest = $path1;
            $smallest = $path2;
        } else {
            $longuest = $path2;
            $smallest = $path1;
        }
        $longuest = self::explode($longuest);
        $smallest = self::explode($smallest);
        $result = [];
        $diffFound = false;
        //print_r('XXX diff from ' . $path1 . ' to ' . $path2 . '<br>');
        //echo('XXX $longuest ' . print_r($longuest, true) . '<br>');
        //echo('XXX $smallest ' . print_r($smallest, true) . '<br>');

        foreach($longuest as $index => $value) {
            //print_r('XXX $diffFound ' . $diffFound . '<br>');
            //print_r('XXX ' . $index . ' : ' . $value . '<br>');
            //print_r('XXX ' . ($value !== $smallest[$index]) . '<br>');
            if (!$diffFound) {
                if (!array_key_exists($index, $smallest) || $value !== $smallest[$index]) {
                    //print_r('DIFF FOUND<br>');
                    $diffFound = true;
                    $result[] = $value;
                }
            } else {
                $result[] = $value;
            }
        }
        //print_r('diff from ' . $path1 . ' to ' . $path2 . ' === ' . implode(GLB_SEPARATOR, $result) . '<br>');
        return implode($file_system ? GLB_SEPARATOR : '/', $result);
    }

    public static function remove_trailing_separator($path) {
        $path = Glb_Text::remove_trailing($path, '/');
        $path = Glb_Text::remove_trailing($path, '\\');
        return $path;
    }

    public static function remove_leading_separator($path) {
        $path = Glb_Text::remove_leading($path, '/');
        $path = Glb_Text::remove_leading($path, '\\');
        return $path;
    }

    /*
     * Concatenate two paths, with the right directory separator
     *
     * @param array $options : can contain 'normalize' => true | false, 'type' => 'file_system' | 'url', 'end_with_separator' => true | false
     *      normalize = true means that all back or forward slashes will be replaced with the occurate slash
     * @return string $full_path
     *
     * @usage
     *      $path = Glb_Path::concat([$path1, $path2, ... $pathn], ['normalize' => true, 'type' => 'url', 'end_with_separator' => true])
     */
    public static function concat($paths, $options = ['normalize' => true, 'type' => 'file_system', 'end_with_separator' => false]) {

        // $paths can also be an array, if not, convert it
        if ( !is_array( $paths ) ) {
            $args = array_slice( func_get_args(), 1 );
        }
        $options = array_merge(['normalize' => true, 'type' => 'file_system', 'end_with_separator' => false], $options);
        $separator = ($options['type'] == 'file_system' ? GLB_SEPARATOR : '/');

        $result = '';
        foreach ($paths as $path) {
            if (!empty($options['normalize'])) {
                $path = self::normalize($path, ($options['type'] == 'file_system'));
            }
            $result .= (empty($result) ? '' : $separator) . self::remove_trailing_separator($path);
        }
        if (!empty($options['end_with_separator'])) {
            $result .= $separator;
        }

        return $result;
    }

    public static function list_folders($path, $pattern = '*', $recursive = false) {
        $path = self::remove_trailing_separator($path);
        if (!$recursive) {
            $results = array_filter(glob($path . GLB_SEPARATOR . $pattern, GLOB_ONLYDIR));
        } else {
            $results = [];
            $folders = glob($path . GLB_SEPARATOR . $pattern, GLOB_ONLYDIR);
            foreach($folders as $key => $value) {
                $results[] = $value;
                // if recursive, then we need to list all folders/sub-folders/sub-sub-folders
                // before applying pattern (wich can be 'sub-sub-folders')
                $results = array_merge($results, self::list_folders($value, '*', $recursive));
            }
            if ($pattern != '*') {
                // convert glob pattern to regex pattern
                $pattern = str_replace('*', '(.*)', $pattern);
                $pattern = str_replace('?', '(.)', $pattern);
                $pattern = '/^' . $pattern . '$/';

                $results = array_filter($results, function($item) use ($pattern) {
                    $parts = self::explode($item);
                    return preg_match($pattern, end($parts));
                });
            }
        }
        return $results;
    }

    public static function list_files($path, $pattern = '*', $recursive = false) {
        $path = self::remove_trailing_separator($path);
        if (!$recursive) {
            $results = array_filter(glob($path . GLB_SEPARATOR . $pattern, GLOB_MARK), function($path) { return is_file($path); });
        } else {
            $results = [];
            // if recursive, then we need to list all folders/sub-folders/sub-sub-folders/file-gloubi.txt
            // before applying pattern (wich can be '*gloubi*')
            $files = glob($path . GLB_SEPARATOR . '*', GLOB_MARK);
            foreach($files as $key => $value) {
                if (!is_dir($value)) {
                    $results[] = $value;
                } else if ($value != "." && $value != "..") {
                    $results = array_merge($results, self::list_files($value, '*', $recursive));
                }
            }
            if ($pattern != '*') {
                // convert glob pattern to regex pattern
                $pattern = str_replace('*', '(.*)', $pattern);
                $pattern = str_replace('?', '(.)', $pattern);
                $pattern = '/^' . $pattern . '$/';

                $results = array_filter($results, function($item) use ($pattern) {
                    $parts = self::explode($item);
                    return preg_match($pattern, end($parts));
                });
            }
        }
        return $results;
    }

    public static function list_all($path, $pattern = '*', $recursive = false) {
        // @todo : could we optimize that ? May be....
        $files = self::list_files($path, $pattern, $recursive);
        $folders = self::list_folders($path, $pattern, $recursive);
        return array_merge($files, $folders);
    }

    /**
     * Copy a folder from $src to $dst (no existency check before)
     * @param string $src
     * @param string $dst
     */
    public static function copy_folder($src, $dst) {
        $dir = opendir($src);
        @mkdir($dst);
        while(false !== ( $file = readdir($dir)) ) {
            if (( $file != '.' ) && ( $file != '..' )) {
                if ( is_dir($src . GLB_SEPARATOR . $file) ) {
                    self::copy_folder($src . GLB_SEPARATOR . $file, $dst . '/' . $file);
                }
                else {
                    copy($src . GLB_SEPARATOR . $file, $dst . GLB_SEPARATOR . $file);
                }
            }
        }
        closedir($dir);
    }

    public static function get_include_content($path, $variables = null) {
        if (!is_file($path)) {
            return false;
        }
        ob_start();
        if (!empty($variables)) {
            extract($variables);
        }
        include $path;
        return ob_get_clean();
    }

    /*
    * Deletes a directory and all its subdirectories recursively
    */
    public static function delete_folder($dir_path) {

        Glb_Log::notice('delete_dir ' . $dir_path);
        if (empty($dir_path)) { return false; }
        if (!file_exists($dir_path))  { return false; }
        $it = new RecursiveDirectoryIterator($dir_path, FilesystemIterator::SKIP_DOTS);
        $files = new RecursiveIteratorIterator($it, RecursiveIteratorIterator::CHILD_FIRST);
        foreach($files as $file) {
            Glb_Log::notice('delete_dir ' . $file);
            if ($file->isDir()){
                rmdir($file->getRealPath());
            } else {
                unlink($file->getRealPath());
            }
        }
        rmdir($dir_path);
    }

    /*
     * function get_dates : get all dates infos
     * @param $path : file / folder path
     * @return array : stat($path)
    */
    public static function get_stat($path, $info = null, $use_cache = true) {

        if (!file_exists($path)) { return false; }

        if (!$use_cache) {
            Glb_Cache::remove(__FILE__ . '::' . __FUNCTION__ . '::' . $path);
        }

        if (Glb_Cache::get(__FILE__ . '::' . __FUNCTION__ . '::' . $path) === null) {
            Glb_Cache::set(__FILE__ . '::' . __FUNCTION__ . '::' . $path, stat($path));
        }

        if ($info === null) {
            return Glb_Cache::get(__FILE__ . '::' . __FUNCTION__ . '::' . $path);
        } else {
            return Glb_Cache::get(__FILE__ . '::' . __FUNCTION__ . '::' . $path)[$info];
        }

    }

    public static function get_path_info($path, $info = null) {
        $path_info = pathinfo($path);

        /*function filePathParts($arg1) {
            echo $arg1['dirname'], "\n";
            echo $arg1['basename'], "\n";
            echo $arg1['extension'], "\n";
            echo $arg1['filename'], "\n";
        }*/

        if ($info === null) {
            return $path_info;
        } else {
            return $path_info[$info];
        }
    }

    public static function last_part($path) {
        $parts = self::explode($path);
        return end($parts);
    }

    public static function sanitize_name($file_name) {
        // Thanks @Łukasz Rysiak!
        $result = mb_ereg_replace('([^\w\s\d\-_~,;\[\]\(\).])', '', trim($file_name));
        // Remove any runs of periods (thanks falstro!)
        return mb_ereg_replace("([\.]{2,})", '', $result);
    }


}