<?php

class Glb_Text
{

    // @var array : default inflector rules
    protected static $_inflector_rules = [
        'plural' => [
            '/(s)tatus$/i'                      => '\1tatuses',
            '/(quiz)$/i'                        => '\1zes',
            '/^(ox)$/i'                         => '\1\2en',
            '/([m|l])ouse$/i'                   => '\1ice',
            '/(matr|vert|ind)(ix|ex)$/i'        => '\1ices',
            '/(x|ch|ss|sh)$/i'                  => '\1es',
            '/([^aeiouy]|qu)y$/i'               => '\1ies',
            '/(hive)$/i'                        => '\1s',
            '/(chef)$/i'                        => '\1s',
            '/(?:([^f])fe|([lre])f)$/i'         => '\1\2ves',
            '/sis$/i'                           => 'ses',
            '/([ti])um$/i'                      => '\1a',
            '/(p)erson$/i'                      => '\1eople',
            '/(?<!u)(m)an$/i'                   => '\1en',
            '/(c)hild$/i'                       => '\1hildren',
            '/(buffal|tomat)o$/i'               => '\1\2oes',
            '/(alumn|bacill|cact|foc|fung|nucle|radi|stimul|syllab|termin)us$/i'
            => '\1i',
            '/us$/i'                            => 'uses',
            '/(alias)$/i'                       => '\1es',
            '/(ax|cris|test)is$/i'              => '\1es',
            '/s$/'                              => 's',
            '/^$/'                              => '',
            '/$/'                               => 's',
        ],
        'singular' => [
            '/(s)tatuses$/i'                    => '\1\2tatus',
            '/^(.*)(menu)s$/i'                  => '\1\2',
            '/(quiz)zes$/i'                     => '\\1',
            '/(matr)ices$/i'                    => '\1ix',
            '/(vert|ind)ices$/i'                => '\1ex',
            '/^(ox)en/i'                        => '\1',
            '/(alias)(es)*$/i'                  => '\1',
            '/(alumn|bacill|cact|foc|fung|nucle|radi|stimul|syllab|termin|viri?)i$/i'
            => '\1us',
            '/([ftw]ax)es/i'                    => '\1',
            '/(cris|ax|test)es$/i'              => '\1is',
            '/(shoe)s$/i'                       => '\1',
            '/(o)es$/i'                         => '\1',
            '/ouses$/'                          => 'ouse',
            '/([^a])uses$/'                     => '\1us',
            '/([m|l])ice$/i'                    => '\1ouse',
            '/(x|ch|ss|sh)es$/i'                => '\1',
            '/(m)ovies$/i'                      => '\1\2ovie',
            '/(s)eries$/i'                      => '\1\2eries',
            '/([^aeiouy]|qu)ies$/i'             => '\1y',
            '/(tive)s$/i'                       => '\1',
            '/(hive)s$/i'                       => '\1',
            '/(drive)s$/i'                      => '\1',
            '/([le])ves$/i'                     => '\1f',
            '/([^rfoa])ves$/i'                  => '\1fe',
            '/(^analy)ses$/i'                   => '\1sis',
            '/(analy|diagno|^ba|(p)arenthe|(p)rogno|(s)ynop|(t)he)ses$/i'
            => '\1\2sis',
            '/([ti])a$/i'                       => '\1um',
            '/(p)eople$/i'                      => '\1\2erson',
            '/(m)en$/i'                         => '\1an',
            '/(c)hildren$/i'                    => '\1\2hild',
            '/(n)ews$/i'                        => '\1\2ews',
            '/eaus$/'                           => 'eau',
            '/^(.*us)$/'                        => '\\1',
            '/s$/i'                             => ''
        ],
        'irregular' => [
            'atlas' => 'atlases',
            'beef' => 'beefs',
            'brief' => 'briefs',
            'brother' => 'brothers',
            'cafe' => 'cafes',
            'child' => 'children',
            'cookie' => 'cookies',
            'corpus' => 'corpuses',
            'cow' => 'cows',
            'criterion' => 'criteria',
            'ganglion' => 'ganglions',
            'genie' => 'genies',
            'genus' => 'genera',
            'graffito' => 'graffiti',
            'hoof' => 'hoofs',
            'loaf' => 'loaves',
            'man' => 'men',
            'money' => 'monies',
            'mongoose' => 'mongooses',
            'move' => 'moves',
            'mythos' => 'mythoi',
            'niche' => 'niches',
            'numen' => 'numina',
            'occiput' => 'occiputs',
            'octopus' => 'octopuses',
            'opus' => 'opuses',
            'ox' => 'oxen',
            'penis' => 'penises',
            'person' => 'people',
            'sex' => 'sexes',
            'soliloquy' => 'soliloquies',
            'testis' => 'testes',
            'trilby' => 'trilbys',
            'turf' => 'turfs',
            'potato' => 'potatoes',
            'hero' => 'heroes',
            'tooth' => 'teeth',
            'goose' => 'geese',
            'foot' => 'feet',
            'foe' => 'foes',
            'sieve' => 'sieves'
        ],
        'invariable' => [
            '.*[nrlm]ese', '.*data', '.*deer', '.*fish', '.*measles', '.*ois',
            '.*pox', '.*sheep', 'people', 'feedback', 'stadia', '.*?media',
            'chassis', 'clippers', 'debris', 'diabetes', 'equipment', 'gallows',
            'graffiti', 'headquarters', 'information', 'innings', 'news', 'nexus',
            'pokemon', 'proceedings', 'research', 'sea[- ]bass', 'series', 'species', 'weather'
        ],
        'transliteration' => [
            'ä' => 'ae',
            'æ' => 'ae',
            'ǽ' => 'ae',
            'ö' => 'oe',
            'œ' => 'oe',
            'ü' => 'ue',
            'Ä' => 'Ae',
            'Ü' => 'Ue',
            'Ö' => 'Oe',
            'À' => 'A',
            'Á' => 'A',
            'Â' => 'A',
            'Ã' => 'A',
            'Å' => 'A',
            'Ǻ' => 'A',
            'Ā' => 'A',
            'Ă' => 'A',
            'Ą' => 'A',
            'Ǎ' => 'A',
            'à' => 'a',
            'á' => 'a',
            'â' => 'a',
            'ã' => 'a',
            'å' => 'a',
            'ǻ' => 'a',
            'ā' => 'a',
            'ă' => 'a',
            'ą' => 'a',
            'ǎ' => 'a',
            'ª' => 'a',
            'Ç' => 'C',
            'Ć' => 'C',
            'Ĉ' => 'C',
            'Ċ' => 'C',
            'Č' => 'C',
            'ç' => 'c',
            'ć' => 'c',
            'ĉ' => 'c',
            'ċ' => 'c',
            'č' => 'c',
            'Ð' => 'D',
            'Ď' => 'D',
            'Đ' => 'D',
            'ð' => 'd',
            'ď' => 'd',
            'đ' => 'd',
            'È' => 'E',
            'É' => 'E',
            'Ê' => 'E',
            'Ë' => 'E',
            'Ē' => 'E',
            'Ĕ' => 'E',
            'Ė' => 'E',
            'Ę' => 'E',
            'Ě' => 'E',
            'è' => 'e',
            'é' => 'e',
            'ê' => 'e',
            'ë' => 'e',
            'ē' => 'e',
            'ĕ' => 'e',
            'ė' => 'e',
            'ę' => 'e',
            'ě' => 'e',
            'Ĝ' => 'G',
            'Ğ' => 'G',
            'Ġ' => 'G',
            'Ģ' => 'G',
            'Ґ' => 'G',
            'ĝ' => 'g',
            'ğ' => 'g',
            'ġ' => 'g',
            'ģ' => 'g',
            'ґ' => 'g',
            'Ĥ' => 'H',
            'Ħ' => 'H',
            'ĥ' => 'h',
            'ħ' => 'h',
            'І' => 'I',
            'Ì' => 'I',
            'Í' => 'I',
            'Î' => 'I',
            'Ї' => 'Yi',
            'Ï' => 'I',
            'Ĩ' => 'I',
            'Ī' => 'I',
            'Ĭ' => 'I',
            'Ǐ' => 'I',
            'Į' => 'I',
            'İ' => 'I',
            'і' => 'i',
            'ì' => 'i',
            'í' => 'i',
            'î' => 'i',
            'ï' => 'i',
            'ї' => 'yi',
            'ĩ' => 'i',
            'ī' => 'i',
            'ĭ' => 'i',
            'ǐ' => 'i',
            'į' => 'i',
            'ı' => 'i',
            'Ĵ' => 'J',
            'ĵ' => 'j',
            'Ķ' => 'K',
            'ķ' => 'k',
            'Ĺ' => 'L',
            'Ļ' => 'L',
            'Ľ' => 'L',
            'Ŀ' => 'L',
            'Ł' => 'L',
            'ĺ' => 'l',
            'ļ' => 'l',
            'ľ' => 'l',
            'ŀ' => 'l',
            'ł' => 'l',
            'Ñ' => 'N',
            'Ń' => 'N',
            'Ņ' => 'N',
            'Ň' => 'N',
            'ñ' => 'n',
            'ń' => 'n',
            'ņ' => 'n',
            'ň' => 'n',
            'ŉ' => 'n',
            'Ò' => 'O',
            'Ó' => 'O',
            'Ô' => 'O',
            'Õ' => 'O',
            'Ō' => 'O',
            'Ŏ' => 'O',
            'Ǒ' => 'O',
            'Ő' => 'O',
            'Ơ' => 'O',
            'Ø' => 'O',
            'Ǿ' => 'O',
            'ò' => 'o',
            'ó' => 'o',
            'ô' => 'o',
            'õ' => 'o',
            'ō' => 'o',
            'ŏ' => 'o',
            'ǒ' => 'o',
            'ő' => 'o',
            'ơ' => 'o',
            'ø' => 'o',
            'ǿ' => 'o',
            'º' => 'o',
            'Ŕ' => 'R',
            'Ŗ' => 'R',
            'Ř' => 'R',
            'ŕ' => 'r',
            'ŗ' => 'r',
            'ř' => 'r',
            'Ś' => 'S',
            'Ŝ' => 'S',
            'Ş' => 'S',
            'Ș' => 'S',
            'Š' => 'S',
            'ẞ' => 'SS',
            'ś' => 's',
            'ŝ' => 's',
            'ş' => 's',
            'ș' => 's',
            'š' => 's',
            'ſ' => 's',
            'Ţ' => 'T',
            'Ț' => 'T',
            'Ť' => 'T',
            'Ŧ' => 'T',
            'ţ' => 't',
            'ț' => 't',
            'ť' => 't',
            'ŧ' => 't',
            'Ù' => 'U',
            'Ú' => 'U',
            'Û' => 'U',
            'Ũ' => 'U',
            'Ū' => 'U',
            'Ŭ' => 'U',
            'Ů' => 'U',
            'Ű' => 'U',
            'Ų' => 'U',
            'Ư' => 'U',
            'Ǔ' => 'U',
            'Ǖ' => 'U',
            'Ǘ' => 'U',
            'Ǚ' => 'U',
            'Ǜ' => 'U',
            'ù' => 'u',
            'ú' => 'u',
            'û' => 'u',
            'ũ' => 'u',
            'ū' => 'u',
            'ŭ' => 'u',
            'ů' => 'u',
            'ű' => 'u',
            'ų' => 'u',
            'ư' => 'u',
            'ǔ' => 'u',
            'ǖ' => 'u',
            'ǘ' => 'u',
            'ǚ' => 'u',
            'ǜ' => 'u',
            'Ý' => 'Y',
            'Ÿ' => 'Y',
            'Ŷ' => 'Y',
            'ý' => 'y',
            'ÿ' => 'y',
            'ŷ' => 'y',
            'Ŵ' => 'W',
            'ŵ' => 'w',
            'Ź' => 'Z',
            'Ż' => 'Z',
            'Ž' => 'Z',
            'ź' => 'z',
            'ż' => 'z',
            'ž' => 'z',
            'Æ' => 'AE',
            'Ǽ' => 'AE',
            'ß' => 'ss',
            'Ĳ' => 'IJ',
            'ĳ' => 'ij',
            'Œ' => 'OE',
            'ƒ' => 'f',
            'Þ' => 'TH',
            'þ' => 'th',
            'Є' => 'Ye',
            'є' => 'ye',
        ]
    ];


    /**
     * Add custom inflection $rules, of either 'plural', 'singular',
     * 'invariable', 'irregular' or 'transliteration' $type.
     *
     * @usage
     *
     * Glb_Text::inflect_rules('plural', ['/^(GloubiKill)or$/i' => '\1ores']);
     * Glb_Text::inflect_rules('singular', ['/^(GloubiKill)ores$/i' => '\1or']);
     * Glb_Text::inflect_rules('irregular', ['Gloubi' => 'Gloubiez']);
     * Glb_Text::inflect_rules('invariable', ['Gloubi']);
     * Glb_Text::inflect_rules('transliteration', ['/å/' => 'aa']);
     *
     * @param string $type The type of inflection, either 'plural', 'singular', 'irregular', 'invariable'
     * or 'transliteration'.
     * @param array $rules Array of rules to be added.
     * @param bool $reset If true, will clears all the rules before importing these new ones
     * @return void
     */
    public static function inflect_rules($type, $rules, $reset = false)
    {
        if ($reset) {
            static::$_inflector_rules[$type] = $rules;
        } else {
            static::$_inflector_rules[$type] = array_merge(
                static::$_inflector_rules[$type],
                $rules
            );
        }
        Glb_Cache::remove('/^Glb_Text(.*)/');
    }

    /**
     * Return $word in plural form.
     *
     * @param string $word Word in singular
     * @return string Word in plural
     *
     * @usage
     *
     *      Glb_Text::pluralize('egg')
     *          returns 'eggs'
     *
     *      Glb_Text::pluralize('variety')
     *          returns 'varieties'
     *
     *      Glb_Text::pluralize('species')
     *          returns 'species'
     *
     *      Glb_Text::pluralize('tomato')
     *          returns 'tomatoes'
     *
     *      Glb_Text::pluralize('house')
     *          returns 'houses'
     */
    public static function pluralize($word)
    {
        // search in cache
        $result = Glb_Cache::get("Glb_Text.pluralize.$word");
        if ($result !== null) { return $result; }

        // search/build cache for irregular pattern
        $irregular = Glb_Cache::get("Glb_Text.irregular.plural", [
            function($rules) {
                return '(?:' . implode('|', array_keys($rules)) . ')';
            } , static::$_inflector_rules['irregular']]);

        // search/build cache for irregular pluralized words
        if (preg_match('/(.*?(?:\\b|_))(' . $irregular . ')$/i', $word, $regs)) {
            return Glb_Cache::set("Glb_Text.pluralize.$word", $regs[1] . substr($regs[2], 0, 1) .
                substr(static::$_inflector_rules['irregular'][strtolower($regs[2])], 1));
        }

        // search/build cache for invariable pattern
        $invariable = Glb_Cache::get("Glb_Text.invariable", [
            function($rules) {
                return '(?:' . implode('|', $rules) . ')';
            } , static::$_inflector_rules['invariable']]);

        // search/build cache for invariable pluralized words
        if (preg_match('/^(' . $invariable . ')$/i', $word, $regs)) {
            return Glb_Cache::set("Glb_Text.pluralize.$word", $word);
        }

        // search/build cache for pluralized words
        foreach (static::$_inflector_rules['plural'] as $rule => $replacement) {
            if (preg_match($rule, $word)) {
                return Glb_Cache::set("Glb_Text.pluralize.$word", preg_replace($rule, $replacement, $word));
            }
        }
    }

    /**
     * Return $word in singular form.
     *
     * @param string $word Word in plural
     * @return string Word in singular
     *
     * @usage
     *      Glb_Text::singularize('eggs') returns 'egg'
     *      Glb_Text::pluralize('varieties') returns 'variety'
     *      Glb_Text::singularize('species') returns 'species'
     *      Glb_Text::singularize('tomatoes') returns 'tomato'
     *      Glb_Text::singularize('houses') returns 'house'
     */
    public static function singularize($word)
    {
        //search in cache
        $result = Glb_Cache::get("Glb_Text.singularize.$word");

        if ($result !== null) { return $result; }

        //search/build cache for irregular pattern
        $irregular = Glb_Cache::get("Glb_Text.irregular.singular", [
            function($rules) {
                return '(?:' . implode('|', $rules) . ')';
            }, static::$_inflector_rules['irregular']]);

        //search/build cache for irregular singularized words
        if (preg_match('/(.*?(?:\\b|_))(' . $irregular . ')$/i', $word, $regs)) {
            return Glb_Cache::set("Glb_Text.singularize.$word", $regs[1] . substr($regs[2], 0, 1) . substr(array_search(strtolower($regs[2]), static::$_inflector_rules['irregular']), 1));
        }

        // search/build cache for invariable pattern
        $invariable = Glb_Cache::get("Glb_Text.invariable", [
            function($rules) {
                return '(?:' . implode('|', $rules) . ')';
            }, static::$_inflector_rules['invariable']]);

        // search/build cache for invariable singularized words
        if (preg_match('/^(' . $invariable . ')$/i', $word, $regs)) {
            return Glb_Cache::set("Glb_Text.singularize.$word", $word);
        }

        // search/build cache for singularized words
        foreach (static::$_inflector_rules['singular'] as $rule => $replacement) {
            if (preg_match($rule, $word)) {
                return Glb_Cache::set("Glb_Text.singularize.$word", preg_replace($rule, $replacement, $word));
            }
        }

        return $word;

    }

    public static function quote_label($label) {
        return "« $label »";
    }

    /**
     * Camelize a string
     *
     * @param string $string String to camelize, can be what you want
     * @param string $separator String to use for splitting words
     * @return string ExampleCamelizedString.
     *
     * @usage
     *      Glb_Text::camelize('this_string_is-a-string') returns 'ThisStringIsAString'
     */
    public static function camelize($string, $separator = '')
    {
        $result = Glb_Cache::get("Glb_Text.camelize.$string");
        if ($result !== null) { return $result; }
        return Glb_Cache::set("Glb_Text.camelize.$string", str_replace(' ', $separator, str_replace(' ', $separator, static::humanize($string))));
    }

    /**
     * Underscore a string
     *
     * @param string $string Any string you want
     * @return string Underscorized version of the input string
     *
     * @usage
     *      Glb_Text::camelize('This string is - a String') returns 'This_string_is_a_String'
     */
    public static function underscore($string)
    {
        $result = Glb_Cache::get("Glb_Text.underscore.$string");
        if ($result !== null) { return $result; }
        return Glb_Cache::set("Glb_Text.underscore.$string", static::delimit($string, '_'));
    }

    /**
     * Convert a string to a dashed-string
     *
     * @param string $string The string to dasherize.
     * @return string Dashed version of the input string
     *
     * @usage
     *      Glb_Text::dasherize('This string isA Big$ _String') returns 'this-string-is-a-big-string'
     */
    public static function dasherize($string) {
        return static::delimit($string, '-');
    }

    /**
     *
     * Convert a string to 'A Human Readable String'.
     *
     * @param string $string String to be humanized
     * @return string 'Human Readable String'
     *
     * @usage
     *     Glb_Text::humanize('This string isA Big$ _String') returns 'This String Is A Big String'
     */
    public static function humanize($string)
    {

        $result = Glb_Cache::get("Glb_Text.humanize.$string");
        if ($result !== null) { return $result; }

        $result = explode(' ', self::delimit($string, ' '));
        foreach ($result as &$word) {
            $word = mb_strtoupper(mb_substr($word, 0, 1)) . mb_substr($word, 1);
        }
        $result = implode(' ', $result);
        return Glb_Cache::set("Glb_Text.humanize.$string", $result);

    }

    /**
     * Convert string to a delimited string
     *
     * @param string $string String to delimit
     * @param string $delimiter the character to use as a delimiter.
     * @return string delimited string
     *
     * @usage
     *
     * Glb_Text::delimit('This is my - string WithALot of$$$ Things inside', '_')
     *      returns 'this_is_my_string_with_a_lot_of_things_inside'
     */
    public static function delimit($string, $delimiter = '-')
    {
        return Glb_Cache::get("Glb_Text.delimit.$delimiter.$string", [
            function() use ($string, $delimiter) {
                // remove all non alphanumeric chars
                $string = preg_replace('/[^\da-z]/i', $delimiter, $string);
                // split camel cased strings
                $string = mb_strtolower(preg_replace('/(?<=\\w)([A-Z])/', $delimiter . '\\1', $string));
                // remove duplicates
                while (mb_strpos($string, $delimiter . $delimiter) !== false) {
                    $string = str_replace($delimiter . $delimiter, $delimiter, $string);
                }
                return $string;
            }
        ]);
    }

    /**
     * Returns a string with all spaces converted to dashes (by default), accented
     * characters converted to non-accented characters, and non word characters removed.
     *
     * @param string $string The string you want to slug
     * @param string $replacement Words glue
     * @return string
     *
     * @usage
     *
     *      Glb_Text::dasherize('This is my - string WithALot of$#^$$ Things inside');
     *          returns "this-is-my-string-with-a-lot-of-things-inside"
     */
    public static function slug($string, $replacement = '-')
    {

        $result = Glb_Cache::get("Glb_Text.slug.$replacement.$string");
        if ($result !== null) { return $result; }

        $quotedReplacement = preg_quote($replacement, '/');
        $map = [
            '/[^\s\p{Zs}\p{Ll}\p{Lm}\p{Lo}\p{Lt}\p{Lu}\p{Nd}]/mu' => ' ',
            '/[\s\p{Zs}]+/mu' => $replacement,
            sprintf('/^[%s]+|[%s]+$/', $quotedReplacement, $quotedReplacement) => '',
        ];

        $result = str_replace(
            array_keys(static::$_inflector_rules['transliteration']),
            array_values(static::$_inflector_rules['transliteration']),
            $string
        );

        $result = self::delimit($result, $replacement);
        $result = strtolower(preg_replace(array_keys($map), array_values($map), $result));
        return Glb_Cache::set("Glb_Text.slug.$replacement.$string", $result);

    }

    /**
     * Determine if a string ends with another string
     *
     * @param string $haystack
     * @param string|array $needle
     * @return bool
     */
    public static function ends_with($haystack, $needle)
    {
        if (is_array($needle)) {
            foreach($needle as $needleItem) {
                if (self::ends_with($haystack, $needleItem)) {
                    return true;
                }
            }
            return false;
        }

        $length = strlen($needle);
        if ($length == 0) { return false; }
        return (substr($haystack, -$length) === $needle);
    }

    /**
     * Determine if a string contains another string
     *
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    public static function contains($haystack, $needle)
    {
        if (empty($haystack) || empty($needle)) {
            return false;
        }
        return (mb_strpos($haystack, $needle) !== false);
    }

    /**
     * Remove the trailing $needle in $haystack if $haystack ends with $needle
     *
     * @param string $haystack
     * @param string $needle
     * @return string
     */
    public static function remove_trailing($haystack, $needle) {
        if (self::ends_with($haystack, $needle)) {
            return substr($haystack, 0, -strlen($needle));
        }
        return $haystack;
    }

    /**
     * Add the trailing $needle in $haystack if $haystack does not end with $needle
     * @param string $haystack
     * @param string $needle
     * @return string
     */
    public static function add_trailing($haystack, $needle) {
        if (!self::ends_with($haystack, $needle)) {
            return $haystack . $needle;
        }
        return $haystack;
    }

    /**
     * Add the leading $needle in $haystack if $haystack does not start with $needle
     * @param string $haystack
     * @param string $needle
     * @return string
     */
    public static function add_leading($haystack, $needle) {
        if (!self::starts_with($haystack, $needle)) {
            return $needle . $haystack;
        }
        return $haystack;
    }

    /**
     * Determine if a string starts with another string
     *
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    public static function starts_with($haystack, $needle) {
        if (is_array($needle)) {
            foreach($needle as $needleItem) {
                if (self::starts_with($haystack, $needleItem)) {
                    return true;
                }
            }
            return false;
        }
        return (substr($haystack, 0, strlen($needle)) === $needle);
    }

    /**
     * Remove the leading $needle in $haystack if $haystack starts with $needle
     *
     * @param string $haystack
     * @param string $needle
     * @return bool
     */
    public static function remove_leading($haystack, $needle) {
        if (self::starts_with($haystack, $needle)) {
            return substr($haystack, strlen($needle));
        }
        return $haystack;
    }

    /**
     * Get the string contained between two other strings and returns only the the first result
     * Will not use Regex, so it will be speeder, man.
     *
     * @param string $string
     * @param string $start_str
     * @param string $end_str
     * @return bool
     */
    public static function between($string, $start_str, $end_str) {
        $start = strpos($string, $start_str);
        if ($start === false) return null;
        $start += strlen($start_str);
        $end = strpos($string, $end_str, $start);
        if ($end === false) return null;
        $length = $end - $start;
        if ($length<=0) return null;
        return substr($string, $start, $length);
    }

    /**
     * Get all the strings contained between two other strings and returns them all
     * Will use Regex, so it will be slower
     *
     * @param string $string
     * @param string $start_str
     * @param string $end_str
     * @return array
     */
    public static function between_all($string, $start_str, $end_str) {
        preg_match_all('/' . preg_quote($start_str) . '(.*?)' . preg_quote($end_str) . '/s', $string, $matches);
        //preg_match_all('/' . $start_str . '(.*?)' . $end_str . '/s', $string, $matches);
        return $matches[1];
    }

    /**
     * Remove the first found instance of string contained between two other strings
     * Will not use Regex, so it will be speeder
     * Will return the removed string, as the $string will be modifed by reference
     *
     * @param string $string
     * @param string $start_str
     * @param string $end_str
     * @return string The removed string
     */
    public static function remove_between(&$string, $start_char, $end_char) {
        $between = self::between($string, $start_char, $end_char);
        if ($between !== null) {
            $string = str_replace($start_char . $between . $end_char, '', $string);
        }
        return $between;
    }

    /**
     * Remove all found strings contained between two other strings
     * Will use Regex, so it will be slower
     * Will return the removed strings as array, as the $string will be modifed by reference
     *
     * @param string $string
     * @param string $start_str
     * @param string $end_str
     * @return array The removed strings
     */
    public static function remove_between_all(&$string, $start_char, $end_char) {
        $between = self::between_all($string, $start_char, $end_char);
        if ($between !== null) {
            foreach($between as $value) {
                $string = str_replace($start_char . $value . $end_char, '', $string);
            }
        }
        return $between;
    }

    /**
     * Ah ! You will laugh. Find a reason to say if a string is a boolean, searching for keywords
     * like 'yes', 'no', 'true', 'false', 'vrai', 'verdadero', 'wahr' etc.
     *
     * @param string $string
     * @param (optional) mixed $default The default result to return if nothing is found
     * @return bool|mixed Returns true/false or $default
     */
    public static function extract_boolean($string, $default = null) {
        $string = mb_strtolower($string);
        if (in_array($string, ['1', 'true', 'vrai', 'verdadero', 'verdadeiro', 'wahr', 'ok', 'yes', 'y', 'oui', 'si', 'ya'])) {
            return true;
        }
        if (in_array($string, ['0', 'false', 'faux', 'falso', 'falsch', 'nok', 'ko', 'no', 'n', 'non', 'nein'])) {
            return false;
        }
        return $default;

    }

    /**
     * Build a unique string ID (something like a GUID) base on time at microsecond and randomization,
     * with optional prefix
     *
     * @param (optional) string $prefix
     * @return string A string that is expected to be absolutely unique in the world... or not.
     */
    public static function unique_id($prefix = null) {
        $m = microtime(true);
        return ($prefix === null ? '' : $prefix) . (sprintf("%8x%05x",floor($m),($m-floor($m))*1000000) . '-' . mt_rand(0, 65355));
    }

    /**
     * Explode a string using break lines chars as delimiters (\r and \n)
     *
     * @param string $string The string to be exploded
     * @return array The found array
     */
    public static function explode_nl($string) {
        $string = str_replace("\r", "\n", $string);
        $string = str_replace("\n\n", "\n", $string);
        return explode("\n", $string);
    }

    /**
     * Concatenate multiple strings with glue(s) if strings are not empty
     *
     * @param string|array $glue The glue that will be added between strings
     * @param mixed ...$texts The texts to concatenate
     *
     * @usage
     *      Glb_Text::concatenate(' : ', 'first', 'second')
     *          returns 'first : second'
     *
     *      Glb_Text::concatenate(' : ', 'first', '', 'third')
     *          returns 'first : third'
     *
     *      Glb_Text::concatenate([':', '::'], 'first', 'second', 'third', 'fourth')
     *          returns 'first:second::third::fourth'
     *
     */
    public static function concatenate($glue, ...$texts) {

        $args = func_get_args();
        array_shift($args);
        Glb_Array::ensure($glue);

        $result = '';
        foreach($args as $index => $arg) {
            if ($arg != '') {
                $sep = $glue[min($index, count($glue)-1)];
                $result = ($result != '' ? $result . $sep . $arg : $arg);
            }
        }

        return $result;

    }

    /**
     * Ensure the content of a string
     *
     * @param string $string
     * @param string $replacement If $string empty
     * @return bool
     */
    public static function ensure($string, $replacement)
    {
        if (empty($string)) {
            return $replacement;
        }
        return $string;
    }

    /**
     * Ensure the content's beginning of a string
     *
     * @param string $string
     * @param string $start If $string doesn't already start with $start
     * @return bool
     */
    public static function ensure_leading($string, $lead)
    {
        if (mb_stripos($string, $lead) !== 0) {
            return $lead . $string;
        }
        return $string;
    }

    /**
     * Ensure the content's beginning of a string
     *
     * @param string $string
     * @param string $start If $string doesn't already start with $start
     * @return bool
     */
    public static function ensure_trailing($string, $trail)
    {
        if (mb_stripos($string, $trail) !== mb_strlen($string) - mb_strlen($trail)) {
            return $string . $trail;
        }
        return $string;
    }


    /*
     * Replace things like {{user.first_name}}, {{user.name}}, [[My translatable string]]
     *  by their real value that must be present in $params variable
     * Fields encapsulated by {{ }} are fields that must be replaced by $params content
     * Fields encapsulated by [[ ]] are fields that need to be replaced by translations
     * @usage Text::replace_params("{{user.first_name}} {{user.last_name}} has : [[a string]]")
     *      returns "Donald Trump has : a string"
     */
    public static function replace_params($text, $params) {
        $replacements = self::between_all($text, '{{', '}}');
        if (!empty($replacements)) {
            foreach($replacements as $replacement) {
                $value = GLb_Hash::get($params, $replacement, '');
                $text = str_replace('{{' . $replacement . '}}', $value, $text);
            }
        }

        $replacements = self::between_all($text, '[[', ']]');
        if (!empty($replacements)) {
            foreach($replacements as $replacement) {
                $value = __glb($replacement);
                $text = str_replace('[[' . $replacement . ']]', $value, $text);
            }
        }


        return str_replace('  ', ' ', $text);
    }

    public static function is_json($string) {
        json_decode($string);
        return (json_last_error() == JSON_ERROR_NONE);
    }
    public static function is_serialized($string) {
        return (@unserialize($string) !== false);
    }

    public static function is_date($date, $format = 'Y-m-d')
    {
        $d = DateTime::createFromFormat($format, $date);
        return $d && $d->format($format) === $date;
    }

}
