<?php
/**
 *
 * Gloubi WP Users Helper
 *
 */

final class Glb_Users
{

    protected static $_cache_wp;
    protected static $_cache_glb;
    protected static $_cache_all = false;

    protected static function _cache_all($refresh = false) {
        if ($refresh || !self::$_cache_all) {
            self::$_cache_wp = (new GLb_Collection(get_users()))->combine('ID', function($item) { return $item; });
            self::$_cache_glb = Glb_Table::get('core_users')->query('select')->execute()->combine('ID', function($item) { return $item; });
            self::$_cache_all = true;
            return true;
        }
        return false;
    }
    /**
     * Load all users and get a Glb_Collection of Glb_Core_User_Entity
     * @param $return
     */
    public static function get_all($refresh = false) {
        self::_cache_all($refresh);
        return self::$_cache_glb;
    }

    /**
     * Load all users and get a Glb_Collection of WP_User
     * @param $return
     */
    public static function get_all_wp($refresh = false) {
        self::_cache_all($refresh);
        return self::$_cache_wp;
    }

    /**
     * Get Glb_Core_User_Entity by ...
     * @param mixed $user Can be a WP_User, a Glb_Core_User_Entity, an id, an array... or whatever you want
     * @param $refresh Force refresh or not ?
     * @return Glb_Core_User_Entity|null
     */
    public static function get($user, $refresh = false)
    {
        if (empty($user)) {
            return null;
        }

        if (!$refresh && is_a($user, 'Glb_Core_User_Entity')) {
            // warning the user could have been partially loaded using ->fields(['ID']) in query for example
            return $user;
        } else if (is_numeric($user)) {
            $user_id = $user;
        } elseif (is_object($user) || is_array($user)) {
            $user_id = Glb_Hash::first($user, ['ID', 'id']);
        } else {
            return null;
        }

        if ($refresh || empty(self::$_cache_glb) || !array_key_exists($user_id, self::$_cache_glb)) {
            self::$_cache_glb[$user_id] = Glb_Table::get('core_users')->query('select')->where(['ID' => $user_id])->execute()->first();
        }
        return self::$_cache_glb[$user_id];
    }


    /**
     * Get WP_User by ...
     * @param mixed $user Can be a WP_User, a Glb_Core_User_Entity, an id, an array... or whatever you want
     * @return WP_User|null
     */
    public static function get_wp($user, $refresh = false)
    {
        if ($user === null) {
            return null;
        }

        if (!$refresh && is_a($user, 'WP_User')) {
            return $user;
        } else if (is_numeric($user)) {
            $user_id = $user;
        } elseif (is_object($user) || is_array($user)) {
            $user_id = Glb_Hash::first($user, ['ID', 'id']);
        } else {
            return null;
        }

        if ($refresh || empty(self::$_cache_wp) || !array_key_exists($user_id, self::$_cache_wp)) {
            self::$_cache_wp[$user_id] = get_user_by('ID', $user_id);
        }
        return self::$_cache_wp[$user_id];
    }

    /**
     * Get user info
     *
     * @param int|null $user_id
     * @param string|null $field_name
     * @return string
     */
    public static function get_info($user_id = null, $field_name = null)
    {
        if (empty($user_id)) {
            $user_id = get_current_user_id();
        }
        if ($field_name === null) {
            return get_userdata($user_id);
        } else {
            $user_data = get_userdata($user_id);
            return $user_data->$field_name;
        }
    }

    /**
     * Get user meta
     *
     * @param int|null $user_id
     * @param string|null $field_name
     * @param string|array $default
     * @return void
     */
    public static function get_meta($user_id = null, $key = null)
    {
        if (empty($user_id)) {
            $user_id = get_current_user_id();
        }
        return get_user_meta($user_id, $key, true);
    }


    public static function current_wp()
    {
        return self::get_wp(get_current_user_id());
    }

    public static function current()
    {
        return self::get(get_current_user_id());
    }


    /**
     * Check if a user has one or multiple capabilities
     *
     * @param mixed $user
     * @param string|array $caps
     * @param string $strategy 'all' (checks for all caps) or 'one' (checks for one cap)
     * @return boolean
     */
    public static function has_caps($user, $caps, $strategy = 'all')
    {
        Glb_Array::ensure($caps);
        if ( ($user = self::get($user)) === null ) {
            return false;
        }
        $user = $user->wp_user;
        foreach ($caps as $cap) {
            $has = user_can($user, $cap);
            if (!$has && $strategy == 'all') {
                return false;
            } else if ($has && $strategy == 'one') {
                return true;
            }
        }
        return ($strategy == 'all');
    }

    /**
     * Get all WP roles that include $included_strategy ('one' or 'all') $included capabilities,
     *      but NOT $excluded_strategy ('one' or 'all') $excluded capabilities
     * Why use this function ? Because Wordpress assigns roles to users, assigns capabilities to roles,
     *      but NEVER assigns directly capabilities to users !
     * @param array|string $included Capability/capabilities required
     * @param array|string $excluded Capability/capabilities excluded
     * @param string $included_strategy 'all' or 'one'
     * @param string $excluded_strategy 'all' or 'one'
     * @return array All roles resolving the $included/$excluded equation
     */
    public static function get_roles_by_caps($included, $excluded = [], $included_strategy = 'all', $excluded_strategy = 'all') {
        if (empty($included)) { return []; }
        Glb_Array::ensure($included, $excluded);

        $roles = (new WP_Roles())->role_objects;
        $filtered = [];
        //glb_dump('get_roles_by_caps inc ' . print_r($included, true) . ' exc ' . print_r($excluded, true));

        foreach($roles as $role_key => $role) {

            $elected = ($included_strategy == 'all');
            foreach($included as $cap) {
                $has = $role->has_cap($cap);
                //glb_dump("$role_key inc has $has");
                if (!$has && $included_strategy == 'all') {
                    $elected = false;
                    break;
                } else if ($has && $included_strategy == 'one') {
                    $elected = true;
                    break;
                }
            }
            /*if ($elected) {
                glb_dump('$role_key ' . $role_key . ' elected by inclusion');
            } else {
                glb_dump('$role_key ' . $role_key . ' not elected by inclusion');
            }*/
            if (!$elected) { continue; }


            if ($excluded) {
                $elected = ($excluded_strategy == 'all');
                foreach($excluded as $cap) {
                    $has = $role->has_cap($cap);
                    if ($has && $excluded_strategy == 'all') {
                        $elected = false;
                        break;
                    } else if (!$has && $excluded_strategy == 'one') {
                        $elected = true;
                        break;
                    }
                }
            }

            /*if ($elected) {
                glb_dump('$role_key ' . $role_key . ' elected by exclusion');
            } else {
                glb_dump('$role_key ' . $role_key . ' not elected by exclusion');
            }*/

            if ($elected) {
                $filtered[$role_key] = $role;
            }
        }
        return $filtered;
    }

    /**
     * If the current user has one/multiple capabilities (if multiple, then checks for ALL caps)
     *
     * @param string|array $caps
     * @return boolean
     */
    public static function current_has_caps($caps, $strategy = 'all')
    {
        return self::has_caps(self::current(), $caps, $strategy);
    }

    /**
     * Check if a user has one or multiple roles
     *
     * @param mixed $user
     * @param string|array $roles
     * @param string $operator 'all' (checks for all roles) or 'one' (checks for one role)
     * @return boolean
     */
    public static function has_roles($user, $roles, $operator = 'all')
    {
        Glb_Array::ensure($roles);
        $user = self::get_wp($user);
        $user_roles = (array)$user->roles;
        if (empty($user_roles)) {
            return false;
        }

        foreach ($roles as $role) {
            $has = in_array($role, $user_roles);
            if (!$has && $operator == 'all') {
                return false;
            } else if ($has && $operator == 'one') {
                return true;
            }
        }
        return ($operator == 'all');
    }

    /**
     * If the current user has one or multiple roles (if multiple, then checks for ALL roles)
     *
     * @param int|object $user
     * @param string|array $caps
     * @param string $operator 'all' | 'one'
     * @return boolean
     */
    public static function current_has_roles($roles, $operator = 'all')
    {
        return self::has_roles(self::current(), $roles, $operator);
    }


    /**
     * Set meta for the $user
     *
     * @param int|object $user Can be an ID or a WP_User
     * @param string $key
     * @param mixed $value
     * @return int|bool Meta ID if the key didn't exist, true on successful update, false on failure.
     */
    public static function set_meta($user, $key, $value)
    {
        if (is_object($user)) {
            $user = Glb_Hash::first($user, ['ID', 'id']);
        }
        return update_user_meta($user, $key, $value);
    }

    /**
     * Get a Glb_Query of users selected from $property / $value
     * @param $property
     * @param $pattern
     */
    public static function get_by($property, $pattern) {

        $users_props = ['ID', 'user_login', 'user_nicename', 'user_email', 'user_url', 'user_registered',
            'user_activation_key', 'user_status', 'display_name'];

        $users_metas = ['nickname', 'first_name', 'last_name', 'description', 'locale'];

        if (in_array($property, ['caps', 'cap', 'capability', 'capabilities' ])) {
            return self::get_by_caps($pattern);
        }

        if (in_array($property, ['roles', 'role'])) {
            return self::get_by_roles($pattern);
        }

        if (in_array($property, $users_props)) {
            $users = Glb_Table::get('core_users')->query('select')
                ->where(["core_users.$property REGEXP" => $pattern]);
            return $users;
        }

        if (in_array($property, $users_props)) {
            $users = Glb_Table::get('core_users')->query('select')
                ->where(["core_users.$property REGEXP" => $pattern]);
            return $users;
        }

        if (in_array($property, $users_metas)) {
            $users = Glb_Table::get('core_users')->query('select')
                ->matching(['usermeta' => ['conditions' => ["usermeta.meta_key" => $property, "usermeta.meta_value REGEXP" => $pattern]]]);
            return $users;
        }

        return null;

    }

    /**
     * Get a Glb_Query containing the request for loading users according to their roles
     *  A Glb_Query so that you can use it for paging later
     *
     * @param string|array $included_caps The capability, or a list of capabilities that must have the users
     * @param string|array $excluded_caps The capability, or a list of capabilities that must NOT have the users
     * @param string $included_operator Not efficient if $included_caps is a string. It can be 'all' | 'one' : if 'all', all the caps are required, if 'one', only one.
     * @param string $excluded_operator Not efficient if $excluded_caps is a string. It can be 'all' | 'one' : if 'all', all the caps must be absent, if 'one', only one.
     * @return Glb_Query of elected users
     */
    public static function get_by_caps($included_caps, $excluded_caps = [], $included_strategy = 'all', $excluded_strategy = 'all')
    {
        $roles = self::get_roles_by_caps($included_caps, $excluded_caps, $included_strategy, $excluded_strategy);
        return self::get_by_roles(array_keys($roles), null, 'one');
    }

    /**
    * Get a Glb_Query containing the request for loading users according to their roles
     *  A Glb_Query so that you can use it for paging later
     *
     * @param string|array $included_roles The role, or a list of roles that must have the users
     * @param string|array $excluded_roles The role, or a list of roles that must NOT have the users
     * @param string $included_operator Not efficient if $included_roles is a string. It can be 'all' | 'one' : if 'all', all the roles are required, if 'one', only one is.
     * @param string $excluded_operator Not efficient if $excluded_roles is a string. It can be 'all' | 'one' : if 'all', all the roles must be absent, if 'one', only one is.
     * @return Glb_Query of elected users
     *
     * @usage :
     *
     *     Glb_Users::get_by_roles(GLB_DOCX_ADMIN_ROLE, GLB_DOCX_USER_ROLE)->all();
     *      returns all users having GLB_DOCX_ADMIN_ROLE but not GLB_DOCX_USER_ROLE
     *
    */
    public static function get_by_roles($included_roles, $excluded_roles = [], $included_operator = 'all', $excluded_operator = 'all')
    {
        if (empty($included_roles)) { return []; }
        Glb_Validator::check_value($excluded_roles, 'empty', []);
        Glb_Array::ensure($included_roles, $excluded_roles);

        $conditions = [];
        if ($included_operator == 'all') {
            foreach($included_roles as $role) {
                // str_repeat because keys must all be different !
                $conditions['capabilities.meta_value REGEXP' . str_repeat(' ', count($conditions))] = '"(' . preg_quote($role) . ')";b:1;';
            }
        } else {
            $included_roles = array_map(function($item) { return preg_quote($item); }, $included_roles);
            $conditions['capabilities.meta_value REGEXP'] = '"(' . implode('|', $included_roles) . ')";b:1;';
        }

        if (!empty($excluded_roles)) {
            if ($excluded_operator == 'all') {
                foreach ($excluded_roles as $role) {
                    // str_repeat because keys must all be different !
                    $conditions['capabilities.meta_value NOT REGEXP' . str_repeat(' ', count($conditions))] = '"(' . preg_quote($role) . ')";b:1;';
                }
            } else {
                $excluded_roles = array_map(function ($item) {
                    return preg_quote($item);
                }, $excluded_roles);
                $conditions['capabilities.meta_value NOT REGEXP'] = '"(' . implode('|', $excluded_roles) . ')";b:1;';
            }
        }

        // The query
        $users = Glb_Table::get('core_users')->query('select')
            ->matching(['capabilities' => ['conditions' => $conditions]]);

        return $users;

    }


    /*
     * Create a new naked Wordpress user, with absolutely no role / capability
     * @param $data An associative array of weird things like user_login, user_email...
     * @return WP_Error | WP_User | null Null should never be returned, please call 911 if you experiment it !
     */
    public static function create_user($data)
    {
        $user = wp_create_user($data['user_login'], '', $data['user_email']);
        if (is_wp_error($user)) {
            return $user;
        }
        if (filter_var($user, FILTER_VALIDATE_INT)) {
            $user = self::get_wp($user);
            foreach ($user->roles as $role) {
                $user->remove_role($role);
            }
            return $user;
        }
        return null;
    }

    /*
     * Update user info and create it if doesn't exist (first_name, last_name, email, website, language...)
     * @param int | WP_User | array | string User object, array or user_id or '~new'
     * @param array $data Associative array with key = data to update, value = value to be set
     * @param array $metas List of meta to take into account (will be removed if not a key of $data, added otherwise)
     * @param array $roles List of roles to take into account (will be removed if not a key of $data, added otherwise)
     * @return bool|WP_User|WP_Error False if something went wrong but we don't know why, WP_Error if we know why, and WP_User if OK
     */
    public static function update_userX($user, $data, $metas = [], $roles = [])
    {
        if ($user === '~new') {
            $user = self::create_user($data);
            if (is_wp_error($user)) {
                return $user;
            }
            if (empty($user->ID)) {
                return false;
            } else {
                $user_id = $user->ID;
            }
        } else {
            $user = self::get($user);
            $user_id = $user->ID;
        }

        $core = ['user_login', 'user_nicename', 'user_status', 'user_email', 'user_url', 'first_name', 'last_name', 'description'];
        $metas = array_merge($metas, ['locale']);

        $core_data = array_intersect_key($data, array_flip($core));
        $meta_data = array_intersect_key($data, array_flip($metas));

        $result = wp_update_user(array_merge($core_data, ['ID' => $user_id]));
        if (is_wp_error($result)) {
            return $result;
        }
        Glb_Log::notice('Updating user result : ', $result);

        // second, save meta data
        foreach ($meta_data as $meta_key => $meta_val) {
            update_user_meta($user_id, $meta_key, $meta_val);
        }

        foreach ($roles as $role) {
            if (!empty($data[$role])) {
                $user->add_role($role);
            } else {
                $user->remove_role($role);
            }
        }

        return self::get_by_id($user_id);
    }

    /*
     * Finally/definitely delete the wordpress user
     * @param int | object $user Can be user_id or WP_User object
     * @return true | false If something wrong happened
     */
    public static function delete_user($user)
    {
        $user = self::get_wp($user);
        if (!empty($user)) {
            // see U later, man
            return wp_delete_user($user->ID);
        }
        return false;
    }

    protected static function _glb_display_format($user, $text)
    {
        $user_info = get_userdata($user->ID);
        if (empty($user_info)) {
            return '';
        }
        $caps = array_keys(array_filter($user_info->allcaps));

        $params = [
            'user_login' => $user_info->user_login,
            'user_nicename' => $user_info->user_nicename,
            'user_url' => $user_info->user_url,
            'user_email' => $user_info->user_email,
            'user_registered' => $user_info->user_registered,
            'display_name' => $user_info->display_name,
            'nickname' => $user_info->nickname,
            'first_name' => $user_info->first_name,
            'last_name' => $user_info->last_name,
            'description' => $user_info->description,
            'id' => $user_info->ID,
            'roles' => implode(', ', $user_info->roles),
            'capabilities' => implode(', ', $caps),
        ];

        return Glb_Text::replace_params($text, $params);
    }

    protected static function get_id($user) {
        if (filter_var($user, FILTER_VALIDATE_INT)) {
            return $user;
        } else {
            return Glb_Hash::first($user, ['ID', 'id'], 'value', 0);
        }
    }

    public static function glb_display($user, $plugin = null, $display_format = null)
    {

        // first, try on the caller plugin
        if ($display_format === null) {
            if ($plugin === null) {
                $plugin = Glb_Plugin::get_caller_plugin();
            } else {
                $plugin = Glb_Plugin::get($plugin);
            }
            $display_format = $plugin->settings->get_value('user_display');
        }
        // second, try on the core plugin
        if ($display_format === null) {
            $plugin = Glb_Plugin::get('glb-core');
            $display_format = $plugin->settings->get_value('user_display');
        }
        if (empty($display_format)) {
            return '';
        }

        $user_id = self::get_id($user);
        if (!empty($user_id)) {
            $result = Glb_Cache::get(__CLASS__ . '::' . __FUNCTION__ . '::' . $display_format . '::' . $user_id);
            if ($result !== null) {
                return $result;
            }
        }
        //glb_dump($user);
        $user = self::get($user);
        //glb_dump($user);

        if (empty($display_format)) {
            return '';
        } else {
            return Glb_Cache::set(__CLASS__ . '::' . __FUNCTION__ . '::' . $display_format . '::' . $user_id,
                self::_glb_display_format($user, $display_format));
        }
    }

    public static function check_current_caps($included_caps, $excluded_caps = [], $exit = true, $included_strategy = 'all', $excluded_strategy = 'all')
    {
        $elected = self::current_has_caps($included_caps, $included_strategy);
        if ($elected) {
            $elected = !self::current_has_caps($excluded_caps, $excluded_strategy);
        }
        if (!$elected && $exit) {
            Glb_Request::instance()->set_error(403, 'redirect:referer', true);
        }
        return $elected;
    }

    public static function check_current_roles($included_roles, $excluded_roles = [], $exit = true, $included_strategy = 'all', $excluded_strategy = 'all')
    {
        $elected = self::current_has_roles($included_roles, $included_strategy);
        if ($elected) {
            $elected = !self::current_has_caps($excluded_roles, $excluded_strategy);
        }
        if (!$elected && $exit) {
            Glb_Request::instance()->set_error(403, 'redirect:referer', true);
        }
        return $elected;
    }
}
