<?php


class Glb_Docx_Repos
{

    /**
     * Check current user capabilities
     * @param $exit Exit if user has no docx caps
     * @return bool|string False if current user has no Docx cap,
     *      the most elevated permission if current user has a Docx cap
     */
    protected static function _check_caps($exit = true)
    {
        if (GLb_Users::current_has_caps(GLB_DOCX_ADMIN_CAPABILITY)) {
            return GLB_DOCX_ADMIN_CAPABILITY;
        }
        if (GLb_Users::current_has_caps(GLB_DOCX_USER_CAPABILITY)) {
            return GLB_DOCX_USER_CAPABILITY;
        }
        if ($exit) {
            Glb_Request::instance()->set_error(403, 'redirect:referer', true);
        }
        return false;
    }

    protected static function _check_user_cap($exit = true) {
        if (GLb_Users::current_has_caps(GLB_DOCX_USER_CAPABILITY)) {
            return GLB_DOCX_USER_CAPABILITY;
        }
        if ($exit) {
            Glb_Request::instance()->set_error(403, 'redirect:referer', true);
        }
        return false;
    }

    protected static function _check_admin_cap($exit = true) {
        if (GLb_Users::current_has_caps(GLB_DOCX_ADMIN_CAPABILITY)) {
            return GLB_DOCX_ADMIN_CAPABILITY;
        }
        if ($exit) {
            Glb_Request::instance()->set_error(403, 'redirect:referer', true);
        }
        return false;
    }


    /*public static function enrich_permissions(&$repo)
    {
        $cap = self::_check_caps(false);

        if (!$cap) {
            return $repo;
        } elseif ($cap == GLB_DOCX_ADMIN_CAPABILITY) {
            $permissions = self::get_permissions($repo->id, '*');
            $repo->permissions = $permissions;
            $repo->my_permission = 'rw';
        } elseif ($cap == GLB_DOCX_USER_CAPABILITY) {
            // if capability is access, then returns its own permission in attribute 'permissions'
            $permissions = self::get_permissions($repo->id, get_current_user_id());
            $repo->my_permission = $permissions;
        }
        return $repo;
    }*/

    protected static function _finalize_query(&$query, $caps) {
        // add users permissions for admins
        if ($caps == GLB_DOCX_ADMIN_CAPABILITY) {
            $query->contain('core_users');
        } else {
            // limit access to the allowed repos for users
            $query->matching(['docx_permissions', ['docx_permissions.user_id' => get_current_user_id()]]);
        }
        // limit to enabled repos from front
        if (Glb_Request::instance()->is_from('public')) {
            $query->where(['docx_repos.status' => 1]);
        }
        $query->order('docx_repos.name');
    }

    /**
     * Get all repositories
     * @return array Of repository Glb_Entity
     */
    public static function get_all($use_cache = true)
    {
        $caps = self::_check_caps(true);
        $cache_key = __FILE__ . '::' . __FUNCTION__ . '::' . get_current_user_id();

        if ($use_cache) {
            $cached = Glb_Cache::get($cache_key);
            if ($cached !== null) {
                return $cached;
            }
        }
        
        $repos = Glb_Table::get('docx_repos')->query('select');
        self::_finalize_query($repos, $caps);
        $repoList = $repos->execute()->combine('folder', function($item) { return $item; });
        Glb_Log::info('$repos ', $repos->sql());
        Glb_Log::info('$repos ', $repoList);
        return Glb_Cache::set($cache_key, $repoList);

/*
        // all repositories for admin
        if ($caps == GLB_DOCX_ADMIN_CAPABILITY) {
            
            $repos = (new Glb_Db_Collection('glb_docx_repos'))
                ->load()->order('name');

            foreach ($repos as $repo) {
                self::enrich_permissions($repo);
                self::enrich_path($repo);
                $result[$repo->folder] = $repo;
            };
        }

        // only the allowed ones for simple users
        elseif (current_user_can(GLB_DOCX_USER_CAPABILITY)) {
            $perms = new Glb_Db_Collection('glb_docx_permissions');
            $perms = $perms->load('user_id = %d', [get_current_user_id()]);
            foreach ($perms as $perm) {
                $repo = (new Glb_Db_Collection('glb_docx_repos'))
                    ->load('status = 1 and id = %d', [$perm->repo_id])->first();

                if ($repo) {
                    $repo->my_permission = $perm->permission;
                    //self::enrich_permissions($repo);
                    self::enrich_path($repo);
                    $result[$repo->folder] = $repo;
                }
            }
            // systematically order by folder name
            ksort($result);
        }*/

            Glb_Cache::set($cache_key, $result);

        return Glb_Cache::get($cache_key);
    }
    
    /*public static function enrich_path(&$repo)
    {
        if (Glb_Hash::is_empty($repo, 'full_path')) {
            $repo->full_path = self::full_path($repo->folder);
        }
        return $repo;
    }*/

    /*
     * Get repositories infos
     * @param string $attribute The attribute in wich to search for $value
     * @param int|string $value The value to search
     * @return array|Glb_Entity
     */
    public static function get_by($attribute, $value)
    {
        $caps = self::_check_caps();
        $repos = Glb_Table::get('docx_repos')->query('select')->where(["docx_repos.$attribute" => $value]);
        self::_finalize_query($repos, $caps);
        return $repos->execute()->combine('folder', function($item) { return $item; });
    }

    /*
     * update permissions in database
     * @param int | array | object $repo : repository id or object
     * @param array permissions : [user_id => 'rw'], ...
     */
    public static function update_permissions($repo, $permissions)
    {
        $repo_id = $repo;
        if (is_array($repo) || is_object($repo)) {
            $repo_id = $repo['id'];
        } else {
            $repo = self::get($repo);
        }
        // first delete all permissions
        Glb_Db::instance()->delete('glb_docx_permissions', ['repo_id' => $repo_id], ['%d']);

        // then insert all new permissions
        foreach ($permissions as $user_id => $permission) {
            Glb_Db::instance()->insert('glb_docx_permissions', [
                'repo_id' => $repo_id,
                'user_id' => $user_id,
                'permission' => $permission,
            ]);
        }
        Glb_Db_Log::instance()->log(
            'perm_update',
            GLB_ACTIVITY_TYPE_ADMIN_ACTION,
            [__glbr('Repository permissions modified for repo « %s »'), $repo['name']],
            null,
            'info'
        );
    }

    /*
     * @function update_users_permissions : update users permissions to database
     * @param int | array | object $user : user id or object
     * @param array permissions : [repo_id => 'rw', ...]
     */
    public static function update_users_permissions($user, $permissions)
    {
        self::_check_admin_cap();
        $user = Glb_Users::get($user);

        // first delete old permissions
        Glb_Db::instance()->delete('glb_docx_permissions', ['user_id' => $user->ID], ['%d']);

        // then insert new permissions
        foreach ($permissions as $repo_id => $permission) {
            Glb_Db::instance()->insert('glb_docx_permissions', [
                'repo_id' => $repo_id,
                'user_id' => $user->ID,
                'permission' => $permission,
            ]);
        }

        Glb_Db_Log::instance()->log(
            'perm_update',
            GLB_ACTIVITY_TYPE_ADMIN_ACTION,
            [__glbr('Repository permissions modified for user « %s »'), $user->user_login],
            null,
            'info'
        );
    }

    public static function get_from_hash($hash, $elfinder)
    {
        $result = [];
        if (is_array($hash)) {
            foreach ($hash as $hash_item) {
                $item = self::get_from_hash($hash_item, $elfinder);
                $result[] = $item;
            }
            $result = array_filter($result);
            return $result;
        } else {
            $volume = $elfinder->getVolume($hash);
            if (empty($volume)) {
                return false;
            }

            $result['repo_folder'] = basename($volume->getRootPath());
            $result['full_path'] = $elfinder->realpath($hash);
            $result['relative_path_from_root'] = self::relative_path($result['full_path']);
            $result['relative_path_from_repo'] = Glb_Path::diff($result['repo_folder'], $result['relative_path_from_root']);
            $result['relative_path_for_display'] = '[' . $result['repo_folder'] . ']/' . $result['relative_path_from_repo'];
            $result['base_name'] = basename($result['full_path']);
            return $result;
        }
    }

    /*
     * Get the Docx repository Glb_Entity
     * @param int|Glb_Entity $repo_id_or_entity Can be either the id or the entity
     * @return Glb_Entity
     */
    public static function get($repo_id_or_entity)
    {
        $caps = self::_check_caps();
        $query = Glb_Table::get('docx_repos')->query('select');
        if (empty($repo_id_or_entity) || $repo_id_or_entity == '~new' || $repo_id_or_entity == -1) {
            return Glb_Table::get('docx_repos')->new_entity();
        } else if (filter_var($repo_id_or_entity, FILTER_VALIDATE_INT) !== false) {
            $query->where(['docx_repos.id' => $repo_id_or_entity]);
            self::_finalize_query($query, $caps);
        } else if ($repo_id_or_entity instanceof Glb_Docx_Repo_Entity || $repo_id_or_entity instanceof Glb_Table_Entity
            || $repo_id_or_entity instanceof Glb_Entity || is_array($repo_id_or_entity)) {
            $query->where(['docx_repos.id' => $repo_id_or_entity['id']]);
            self::_finalize_query($query, $caps);
        }

        $repo = $query->execute()->first();
        if (empty($repo['id'])) {
            return null;
        }

        //self::enrich_path($repo);
        //self::enrich_permissions($repo);
        return $repo;
    }

    /*
     * get all permissions for a given repository, and eventually a given user_id.
     * If $user_id is null, returns all permissions for get_current_user_id()
     * If $user_id is '*', returns all permissions for all users
     * If $user_id is integer, returns the permissions for this user on this repo
     *
     * @param int $repo_id : id of the repository
     * @param $user_id : id of the user, or "*" or null.
     * @return array [ ['user_id' => '1', 'permission' => 'rw'], ['user_id' => '3', 'permission' => 'r']]
     *  or string : 'rw' if user_id is an integer or is null
     *  or false if params are not valid
     */
    public static function get_permissionsX($repo_id, $user_id = null)
    {
        if (empty($repo_id)) {
            return false;
        }
        $processed_user_id = 0;

        if ($user_id === '*' or $user_id === 'all') {
            $processed_user_id = '%';
        } elseif (is_null($user_id)) {
            $processed_user_id = get_current_user_id();
        } elseif (is_integer($user_id)) {
            $processed_user_id = $user_id;
        }

        if (empty($processed_user_id) || !is_numeric($repo_id)) {
            return false;
        }

        $items = Glb_Table::get('docx_permissions')->query('select')
            ->where(['repo_id' => $repo_id, 'user_id LIKE' => $processed_user_id])->execute()
            ->combine('user_id', 'permission');

        if (is_integer($processed_user_id)) {
            return (empty($items[$processed_user_id]) ? '' : $items[$processed_user_id]);
        } else {
            return $items;
        }
    }

    /*
     * get all permissions for a given user.
     * @param $user_id : id of the user.
     * @return array [ ['repo_id' => 1, 'permission' => 'rw'], ['repo' => 2, 'permission' => 'r']]
     * or string : 'rw'
     */
    public static function get_users_permissionsX($user_id)
    {
        if (empty($user_id)) {
            return false;
        }
        $perms = new Glb_Db_Collection('glb_docx_permissions');
        $perms = $perms->load('user_id = %d', [$user_id])->values();
        return $perms;
    }

    /*
     * update repository from $_POST data (name, description, and permissions if 'user' is a key
     * @param array $data : ['id' => 2, 'name' => 'repo 1', 'description' => 'desc repo 1', 'users' => [user_id => ['r' => '', 'w' => '']]
     * @return bool
     */
    public static function update_repo($repo, $data)
    {
        self::_check_admin_cap();
        //Glb_Log::notice('update_repo ', $data);
        if ($data['id'] == '~new' && !empty($data['name'])) {
            // create folder
            $data['id'] = Glb_Db::instance()->insert('glb_docx_repos', ['name' => $data['name'], 'description' => $data['description'], 'folder' => $data['folder'], 'status' => (!empty($data['status'] ? 1 : 0)), 'created' => date('Y-m-d H:i:s'), 'modified' => date('Y-m-d H:i:s')], ['%s', '%s', '%s', '%s', '%s']);
            Glb_Db_Log::instance()->log('repo_add', GLB_ACTIVITY_TYPE_ADMIN_ACTION, [__glbr('Repository « %s » created'), $data['name']], null, 'info');
        } else {
            Glb_Db::instance()->update('glb_docx_repos', ['name' => $data['name'], 'description' => $data['description'], 'folder' => $data['folder'], 'modified' => date('Y-m-d H:i:s'), 'status' => (!empty($data['status'] ? 1 : 0))], ['id' => $data['id']], ['%s', '%s', '%s'], ['%d']);
            Glb_Db_Log::instance()->log('repo_update', GLB_ACTIVITY_TYPE_ADMIN_ACTION, [__glbr('Repository « %s » updated'), $data['name']], null, 'info');
        }
        if (!empty($data['users'])) {
            $permissions = array_map(function ($value) {
                return implode('', array_keys(array_filter($value)));
            }, $data['users']);
            $permissions = array_filter($permissions);
        } else {
            $permissions = [];
        }
        self::update_permissions($data['id'], $permissions);
        return self::get($data['id']);
    }

    /*
     * Delete repository and all its files
     * @param array | int $repo : repository_id or $repo object
     * @return bool
    */
    public static function delete_repoX($repo)
    {
        self::_check_admin_cap();

        // log action
        Glb_Db_Log::instance()->log('repo_delete', GLB_ACTIVITY_TYPE_ADMIN_ACTION, [__glbr('Repository « %s » deleted'), $repo->name], null, 'info');

        // @todo : set to trash and remove it after a long time
        $repo = self::get($repo);

        // delete folder
        Glb_Path::delete_folder(self::full_path($repo->folder));

        // delete repo in database
        Glb_Db::instance()->delete('glb_docx_repos', ['id' => $repo->id], ['%d']);

        // delete associated permissions
        Glb_Db::instance()->delete('glb_docx_permissions', ['repo_id' => $repo->id], ['%d']);

        return true;
    }

    /*
     * returns the full path from the folder name
     */
    public static function full_path($folder)
    {
        $root_path = Glb_Plugin::get_registered('glb-docx')->get_config('root_path');
        return Glb_Path::concat([$root_path, $folder]);
    }

    /*public static function extract_attributes($attribute)
    {
        $repos = self::get_all();
        $result = [];
        foreach ($repos as $repo_key => $repo) {
            $result[$repo->id] = $repo->$attribute;
        }
        return $result;
    }*/

    /*
     * returns the relative path from the root repo folder
     */
    public static function relative_path($folder)
    {
        $root_path = Glb_Plugin::get_registered('glb-docx')->get_config('root_path');
        return Glb_Path::diff($root_path, $folder);
    }

    /*
     * returns the root path
     */
    public static function root_path($with_ending_separator = true)
    {
        return Glb_Plugin::get_registered('glb-docx')->get_config('root_path');
    }

    public static function validate_folder($folder)
    {
        $result = Glb_Path::sanitize_name($folder);
        $i = 1;
        while (file_exists(self::full_path($result))) {
            $result = $folder . ' (' . $i . ')';
            $i++;
        }
        return $result;
    }

    /*
    read file system volumes, and add them to database if doesnt exists in it
    @param : no param
    @return : no return
    */

    public static function synchronize_file_system()
    {
        Glb_Log::notice('synchronize_file_system ');

        $root_path = Glb_Plugin::get_registered('glb-docx')->get_config('root_path');
        $folders = Glb_Path::list_folders($root_path);
        Glb_Log::notice('$folders ', $folders);
        Glb_Db_Log::instance()->log('repo_synchro', GLB_ACTIVITY_TYPE_ADMIN_ACTION, [__glbr('Repositories resynchronized')], null, 'info');
        foreach ($folders as $folder) {
            $pathinfo = pathinfo($folder);
            if (!Glb_Db::instance()->exists('glb_docx_repos', 'folder = %s', [$pathinfo['filename']])) {
                $repo_id = Glb_Db::instance()->insert('glb_docx_repos', ['folder' => $pathinfo['filename'], 'name' => ucfirst($pathinfo['filename']), 'status' => 1, 'created' => date('Y-m-d H:i:s'), 'modified' => date('Y-m-d H:i:s')], ['%s', '%s', '%d', '%s', '%s']);
                Glb_Db_Log::instance()->log('repo_add', GLB_ACTIVITY_TYPE_ADMIN_ACTION, [__glbr('Repository « %s » created by synchro'), ucfirst($pathinfo['filename'])], null, 'info');
            }
        }
    }
}
